<?php
/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * Utility class for checking preconditions
 */

declare(strict_types=1);

namespace Mambo\Util;

class Preconditions
{

   /**
    * Checks if a value is null and throws an exception if it is
    * @param mixed $toCheck Value to check for null
    * @param string $errorMessage Error message for the exception
    * @throws \InvalidArgumentException If $toCheck is null
    */
   public static function checkNotNull($toCheck, string $errorMessage): void
   {
      if ($toCheck === null) {
         throw new \InvalidArgumentException($errorMessage);
      }
   }

   /**
    * Checks if a value is empty and throws an exception if it is
    * @param mixed $toCheck Value to check for emptiness
    * @param string $errorMessage Error message for the exception
    * @throws \InvalidArgumentException If $toCheck is empty
    */
   public static function checkNotEmpty($toCheck, string $errorMessage): void
   {
      if (empty($toCheck)) {
         throw new \InvalidArgumentException($errorMessage);
      }
   }

   /**
    * Checks if a string starts with a specific prefix
    * @param string $toCheck String to check
    * @param string $prefix Prefix to look for
    * @param string $errorMessage Error message for the exception
    * @throws \InvalidArgumentException If $toCheck doesn't start with $prefix
    */
   public static function checkStartsWith(string $toCheck, string $prefix, string $errorMessage): void
   {
      if (!str_starts_with($toCheck, $prefix)) {
         throw new \InvalidArgumentException($errorMessage);
      }
   }
}
