<?php
/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * The MamboWebhooksService class handles all Webhook related requests
 * to the Mambo API.
 */

declare(strict_types=1);

namespace Mambo\Webhook;

use Mambo\Webhook\Data\WebhookRequestData;
use Mambo\Webhook\Param\WebhookGetListParams;
use Mambo\Common\Service\AbstractService;
use Mambo\Common\Data\DeleteRequestData;
use Mambo\Http\ApiUrlBuilder;
use Mambo\Http\RequestOptions;
use Mambo\Http\HttpClient;

class WebhooksService extends AbstractService
{
	/**
	 * Webhook service end webhook URI
	 * @var string
	 */
	const WEBHOOKS_URI = "/v1/webhooks";
	const WEBHOOKS_ID_URI = "/v1/webhooks/{id}";


	/**
	 * This method is used to create a new webhook.
	 *
	 * @param WebhookRequestData $data			The data sent to the API in order to create a webhook
	 * @param RequestOptions $requestOptions		The requestOptions to be used with the request
	 * @return array|object
	 */
	public function create(WebhookRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::WEBHOOKS_URI);
		return $this->httpPost($urlBuilder, $requestOptions, $data);
	}


	/**
	 * Update an existing webhook by ID.
	 * @param string $id					The ID of the webhook to update
	 * @param WebhookRequestData $data	The data with which to update the specified webhook object
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 * @return array|object
	 */
	public function update(string $id, WebhookRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::WEBHOOKS_ID_URI);
		$urlBuilder->withPathId($id);
		return $this->httpPut($urlBuilder, $requestOptions, $data);
	}


	/**
	 * Delete a webhook by it's ID
	 *
	 * @param string $id				The ID of the webhook to delete
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function delete(string $id, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::WEBHOOKS_ID_URI);
		$urlBuilder->withPathId($id);
		return $this->httpDelete($urlBuilder, $requestOptions);
	}


	/**
	 * Delete a list of webhooks by their ID
	 *
	 * @param DeleteRequestData $data	The data with the webhooks to delete
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function deleteWebhooks(DeleteRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::WEBHOOKS_URI);
		return $this->httpDelete($urlBuilder, $requestOptions, $data);
	}


	/**
	 * Get a webhook by it's ID
	 *
	 * @param string $id			The ID of the webhook to retrieve
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function get(string $id, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::WEBHOOKS_ID_URI);
		$urlBuilder->withPathId($id);
		return $this->httpGet($urlBuilder, $requestOptions);
	}


	/**
	 * Get the list of webhooks
	 * 
	 * @param WebhookGetListParams $params		The query parameters to be used with the request
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function getWebhooks(WebhookGetListParams $params = null, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::WEBHOOKS_URI);
		$urlBuilder->withQueryParams($params);

		return $this->httpGet($urlBuilder, $requestOptions);
	}
}
