<?php
/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * This object captures the data required by the Webhook API in
 * order to create / update webhooks.
 */

declare(strict_types=1);

namespace Mambo\Webhook\Data;

use Mambo\Common\Data\RequestData;

class WebhookRequestData implements RequestData
{
	private $data = array();

	/**
	 * The name of the webhook.
	 * See the webhook page in administration panel for more information.
	 * @return string name
	 */
	public function getName()
	{
		return $this->data['name'];
	}
	public function setName($name)
	{
		$this->data['name'] = $name;
	}


	/**
	 * The description for the webhook
	 * See the webhook page in administration panel for more information.
	 * @return string description
	 */
	public function getDescription()
	{
		return $this->data['description'];
	}
	public function setDescription($description)
	{
		$this->data['description'] = $description;
	}


	/**
	 * Indicates whether the webhook is active or inactive
	 * See the webhook page in administration panel for more information.
	 * @return boolean active
	 */
	public function getActive()
	{
		return $this->data['active'];
	}
	public function setActive($active)
	{
		$this->data['active'] = $active;
	}


	/**
	 * The endpoint URL to which the webhook payload should be sent
	 * See the webhook page in administration panel for more information.
	 * @return string endpointUrl
	 */
	public function getEndpointUrl()
	{
		return $this->data['endpointUrl'];
	}
	public function setEndpointUrl($endpointUrl)
	{
		$this->data['endpointUrl'] = $endpointUrl;
	}


	/**
	 * Indicates whether the SSL certificates of the endpoint
	 * should be verified before sending the payload
	 * See the webhook page in administration panel for more information.
	 * @return boolean sslVerification
	 */
	public function getSslVerification()
	{
		return $this->data['sslVerification'];
	}
	public function setSslVerification($sslVerification)
	{
		$this->data['sslVerification'] = $sslVerification;
	}


	/**
	 * The secret is used to sign the payload to ensure it comes from Mambo
	 * See the webhook page in administration panel for more information.
	 * @return string secret
	 */
	public function getSecret()
	{
		return $this->data['secret'];
	}
	public function setSecret($secret)
	{
		$this->data['secret'] = $secret;
	}


	/**
	 * The events which should cause the webhook to deliver the payload.
	 * Valid values include: ActivityCreated, ActivityBehavioursAdded, ActivityAwardedAdded,
	 * ActivityRejected, NotificationCreated, NotificationRead, PointCreated, PointUpdated, PointWalletDebitExpired,
	 * BehaviourCreated, BehaviourUpdated, RewardCreated, RewardUpdated, KPICreated, KPIUpdated, LeaderboardCreated,
	 * LeaderboardUpdated, LeaderboardRankingUpdated, CouponCreated, CouponUpdated, TagCreated, TagUpdated,
	 * UserCreated, UserUpdated, UserMemberJoined, UserMemberLeft
	 * See the webhook page in administration panel for more information.
	 */
	public function getEvents()
	{
		return $this->data['events'];
	}
	public function setEvents(array $events)
	{
		$this->data['events'] = $events;
	}


	/**
	 * The sites from which the webhook should listen for events
	 * See the webhook page in administration panel for more information.
	 */
	public function getSiteIds()
	{
		return $this->data['siteIds'];
	}
	public function setSiteIds(array $siteIds)
	{
		$this->data['siteIds'] = $siteIds;
	}


	/**
	 * Whether Webhook payloads should use the pinned version of the latest
	 * version of Mambo. Valid values are "pinned" and "latest".
	 * See the webhook section in the administrator manual for more information.
	 * @return string payloadVersion
	 */
	public function getPayloadVersion()
	{
		return $this->data['payloadVersion'];
	}
	public function setPayloadVersion($payloadVersion)
	{
		$this->data['payloadVersion'] = $payloadVersion;
	}


	/**
	 * Return the JSON string equivalent of this object
	 */
	public function getJsonString(): string
	{
		$json = $this->data;
		return json_encode($json);
	}
}
