<?php
/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * The MamboUsersService class handles all User related requests
 * to the Mambo API.
 */

declare(strict_types=1);

namespace Mambo\User;

use Mambo\User\Data\UserRequestData;
use Mambo\User\Param\UserBlacklistGetListParams;
use Mambo\User\Param\UserCreateParams;
use Mambo\User\Param\UserCustomFieldParams;
use Mambo\User\Param\UserGetAchievementsParams;
use Mambo\User\Param\UserGetActivitiesParams;
use Mambo\User\Param\UserGetAllAchievementsParams;
use Mambo\User\Param\UserGetAllCouponsParams;
use Mambo\User\Param\UserGetAllGiftsParams;
use Mambo\User\Param\UserGetAllLevelsParams;
use Mambo\User\Param\UserGetAllMissionsParams;
use Mambo\User\Param\UserGetAllRewardsParams;
use Mambo\User\Param\UserGetByIdParams;
use Mambo\User\Param\UserGetCouponsParams;
use Mambo\User\Param\UserGetGiftsParams;
use Mambo\User\Param\UserGetKpisParams;
use Mambo\User\Param\UserGetLeaderboardParams;
use Mambo\User\Param\UserGetLevelsParams;
use Mambo\User\Param\UserGetListParams;
use Mambo\User\Param\UserGetMissionsParams;
use Mambo\User\Param\UserGetNotificationsParams;
use Mambo\User\Param\UserGetParams;
use Mambo\User\Param\UserGetPointTransactionsParams;
use Mambo\User\Param\UserGetPointWalletsParams;
use Mambo\User\Param\UserGetRewardsParams;
use Mambo\User\Param\UserSearchParams;
use Mambo\User\Param\UserUpdateParams;
use Mambo\User\Param\UserUploadParams;
use Mambo\Common\Service\AbstractService;
use Mambo\Common\Data\CustomFieldValueRequestData;
use Mambo\Common\Data\CustomFieldValue;
use Mambo\Http\ApiUrlBuilder;
use Mambo\Http\RequestOptions;
use Mambo\Http\HttpClient;
use Mambo\Util\Preconditions;

class UsersService extends AbstractService
{
	/**
	 * User service end point URIs
	 */
	const USER_ID_URI = "/v1/users/{id}";

	const USER_SITE_URI = "/v1/{site}/users";
	const USER_UUID_URI = "/v1/{site}/users/{uuid}";
	const USER_SEARCH_URI = "/v1/{site}/users/search";
	const USER_IMAGE_URI = "/v1/{site}/users/{uuid}/image";

	const USER_BLACKLIST_URI = "/v1/{site}/users/blacklist";
	const USER_BLACKLIST_UUID_URI = "/v1/{site}/users/blacklist/{uuid}";

	const USER_RESET_URI = "/v1/{site}/users/{uuid}/reset";
	const USER_CUSTOM_URI = "/v1/{site}/users/{uuid}/custom_fields";
	const USER_CUSTOM_IMAGE_URI = "/v1/{site}/users/{uuid}/custom_fields/image";
	const USER_REWARDS_URI = "/v1/{site}/users/{uuid}/rewards";
	const USER_ALL_REWARDS_URI = "/v1/{site}/users/{uuid}/rewards/all";
	const USER_LEVELS_URI = "/v1/{site}/users/{uuid}/rewards/levels";
	const USER_ALL_LEVELS_URI = "/v1/{site}/users/{uuid}/rewards/levels/all";
	const USER_ACHIEVEMENTS_URI = "/v1/{site}/users/{uuid}/rewards/achievements";
	const USER_ALL_ACHIEVEMENTS_URI = "/v1/{site}/users/{uuid}/rewards/achievements/all";
	const USER_MISSIONS_URI = "/v1/{site}/users/{uuid}/rewards/missions";
	const USER_ALL_MISSIONS_URI = "/v1/{site}/users/{uuid}/rewards/missions/all";
	const USER_GIFTS_URI = "/v1/{site}/users/{uuid}/rewards/gifts";
	const USER_ALL_GIFTS_URI = "/v1/{site}/users/{uuid}/rewards/gifts/all";
	const USER_COUPONS_URI = "/v1/{site}/users/{uuid}/coupons";
	const USER_ALL_COUPONS_URI = "/v1/{site}/users/{uuid}/coupons/all";
	const USER_ACTIVITIES_URI = "/v1/{site}/users/{uuid}/activities";
	const USER_KPIS_URI = "/v1/{site}/users/{uuid}/kpis";
	const USER_NOTIFICATIONS_URI = "/v1/{site}/users/{uuid}/notifications";
	const USER_CLEAR_NOTIFICATIONS_URI = "/v1/{site}/users/{uuid}/notifications/clear";
	const USER_LEADERBOARD_URI = "/v1/{site}/users/{uuid}/leaderboards/{id}";
	const USER_POINT_WALLETS_URI = "/v1/{site}/users/{uuid}/points/wallets";
	const USER_POINT_TRANSACTIONS_URI = "/v1/{site}/users/{uuid}/points/transactions/{pointId}";


	/**
	 * Create a new user
	 *
	 * @param string $siteUrl					The site in which you want to create the user
	 * @param UserRequestData $data				The data used to create the user
	 * @param UserCreateParams $params			The query parameters to be used with the request
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function create(
		string $siteUrl,
		UserRequestData $data,
		UserCreateParams $params = null,
		RequestOptions $requestOptions = null
	): array|object {

		$urlBuilder = new ApiUrlBuilder(self::USER_SITE_URI);
		$urlBuilder->withPathSite($siteUrl);
		$urlBuilder->withQueryParams($params);

		return $this->httpPost($urlBuilder, $requestOptions, $data);
	}


	/**
	 * Update an existing user
	 *
	 * @param string $siteUrl					The site to which the user belongs
	 * @param string $uuid						The unique user ID of the user to update
	 * @param UserRequestData $data				The data with which to update the specified user object
	 * @param UserUpdateParams $params			The query parameters to be used with the request
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function update(
		string $siteUrl,
		string $uuid,
		UserRequestData $data,
		UserUpdateParams $params = null,
		RequestOptions $requestOptions = null
	): array|object {

		$urlBuilder = new ApiUrlBuilder(self::USER_UUID_URI);
		$urlBuilder->withPathSite($siteUrl);
		$urlBuilder->withPathUuid($uuid);
		$urlBuilder->withQueryParams($params);

		return $this->httpPut($urlBuilder, $requestOptions, $data);
	}


	/**
	 * Upload an image for a user. This will take precedence over
	 * the user's pictureUrl.
	 *
	 * @param string $siteUrl					The site to which the user belongs
	 * @param string $uuid						The unique user ID of the user to update
	 * @param string|false $image				The path to the image to upload
	 * @param UserUploadParams $params			The query parameters to be used with the request
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function uploadImage(
		string $siteUrl,
		string $uuid,
		string|false $image,
		UserUploadParams $params = null,
		RequestOptions $requestOptions = null
	): array|object {

		Preconditions::checkNotEmpty($image, "The image must not be empty");

		$urlBuilder = new ApiUrlBuilder(self::USER_IMAGE_URI);
		$urlBuilder->withPathSite($siteUrl);
		$urlBuilder->withPathUuid($uuid);
		$urlBuilder->withQueryParams($params);

		return $this->httpUpload($urlBuilder, $image, $requestOptions);
	}


	/**
	 * Get a user
	 *
	 * @param string $siteUrl					The site from which you want to retrieve the user
	 * @param string $uuid						The unique user ID of the user to retrieve
	 * @param UserGetParams $params				The query parameters to be used with the request
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function get(
		string $siteUrl,
		string $uuid,
		UserGetParams $params = null,
		RequestOptions $requestOptions = null
	): array|object {

		$urlBuilder = new ApiUrlBuilder(self::USER_UUID_URI);
		$urlBuilder->withPathSite($siteUrl);
		$urlBuilder->withPathUuid($uuid);
		$urlBuilder->withQueryParams($params);

		return $this->httpGet($urlBuilder, $requestOptions);
	}


	/**
	 * Get a user by ID
	 *
	 * @param string $id						The ID of the user to retrieve
	 * @param UserGetByIdParams $params			The query parameters to be used with the request
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function getById(
		string $id,
		UserGetByIdParams $params = null,
		RequestOptions $requestOptions = null
	): array|object {

		$urlBuilder = new ApiUrlBuilder(self::USER_ID_URI);
		$urlBuilder->withPathId($id);
		$urlBuilder->withQueryParams($params);

		return $this->httpGet($urlBuilder, $requestOptions);
	}


	/**
	 * Delete a user
	 *
	 * @param string $siteUrl	The site from which to delete the user
	 * @param string $uuid		The unique user ID of the user to delete
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function delete(string $siteUrl, string $uuid, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::USER_UUID_URI);
		$urlBuilder->withPathSite($siteUrl);
		$urlBuilder->withPathUuid($uuid);
		return $this->httpDelete($urlBuilder, $requestOptions);
	}


	/**
	 * Reset a user. This will clear all the data associated to the user, except
	 * for the basic data (i.e. username, email, name, etc).
	 *
	 * @param string $siteUrl	The site from which to reset the user
	 * @param string $uuid		The unique user ID of the user to reset
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function reset(string $siteUrl, string $uuid, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::USER_RESET_URI);
		$urlBuilder->withPathSite($siteUrl);
		$urlBuilder->withPathUuid($uuid);
		return $this->httpPost($urlBuilder, $requestOptions);
	}


	/**
	 * Get the list of users for the specified site
	 *
	 * @param string $siteUrl					The site for which to retrieve the list of users
	 * @param UserGetListParams $params			The query parameters to be used with the request
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function getUsers(
		string $siteUrl,
		UserGetListParams $params = null,
		RequestOptions $requestOptions = null
	): array|object {

		$urlBuilder = new ApiUrlBuilder(self::USER_SITE_URI);
		$urlBuilder->withPathSite($siteUrl);
		$urlBuilder->withQueryParams($params);

		return $this->httpGet($urlBuilder, $requestOptions);
	}


	/**
	 * Search the users of the specified site starting from the page
	 * specified and returning the number of users specified by count. Also
	 * filter by the specified field.
	 *
	 * @param string $siteUrl					The site for which to retrieve the list of users.
	 * @param string $query						The search query to be used.
	 * @param UserSearchParams $params			The query parameters to be used with the request
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function searchUsers(
		string $siteUrl,
		string $query,
		UserSearchParams $params = null,
		RequestOptions $requestOptions = null
	): array|object {

		Preconditions::checkNotEmpty($query, "query must not be empty");

		$params = ($params !== null) ? $params : new UserSearchParams();
		$params->searchQuery($query);

		$urlBuilder = new ApiUrlBuilder(self::USER_SEARCH_URI);
		$urlBuilder->withPathSite($siteUrl);
		$urlBuilder->withQueryParams($params);

		return $this->httpGet($urlBuilder, $requestOptions);
	}


	/**
	 * Add a user to the blacklist. The blacklist stops user's from earning points.
	 *
	 * @param string $siteUrl	The site to which the user belongs
	 * @param string $uuid		The unique user ID of the user to be added to the blacklist
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function addToBlacklist(string $siteUrl, string $uuid, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::USER_BLACKLIST_UUID_URI);
		$urlBuilder->withPathSite($siteUrl);
		$urlBuilder->withPathUuid($uuid);
		return $this->httpPost($urlBuilder, $requestOptions);
	}


	/**
	 * Remove a user from the blacklist. The blacklist stops user's from earning points.
	 *
	 * @param string $siteUrl	The site to which the user belongs
	 * @param string $uuid		The unique user ID of the user to be removed from the blacklist
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function removeFromBlacklist(string $siteUrl, string $uuid, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::USER_BLACKLIST_UUID_URI);
		$urlBuilder->withPathSite($siteUrl);
		$urlBuilder->withPathUuid($uuid);
		return $this->httpDelete($urlBuilder, $requestOptions);
	}


	/**
	 * Get the list of blacklisted users for the specified site
	 *
	 * @param string $siteUrl						The site for which to retrieve the list blacklisted of users
	 * @param UserBlacklistGetListParams $params	The query parameters to be used with the request
	 * @param RequestOptions $requestOptions		The requestOptions to be used with the request
	 */
	public function getBlacklistedUsers(
		string $siteUrl,
		UserBlacklistGetListParams $params = null,
		RequestOptions $requestOptions = null
	): array|object {

		$urlBuilder = new ApiUrlBuilder(self::USER_BLACKLIST_URI);
		$urlBuilder->withPathSite($siteUrl);
		$urlBuilder->withQueryParams($params);

		return $this->httpGet($urlBuilder, $requestOptions);
	}


	/**
	 * Get a user's rewards. This returns a list of all the rewards
	 * awarded to the specified user (achievements + levels).
	 *
	 * @param string $siteUrl					The site to which the user belongs
	 * @param string $uuid						The unique user ID of the user who's rewards information is to be retrieved
	 * @param UserGetRewardsParams $params		The query parameters to be used with the request
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function getRewards(
		string $siteUrl,
		string $uuid,
		UserGetRewardsParams $params = null,
		RequestOptions $requestOptions = null
	): array|object {

		$urlBuilder = new ApiUrlBuilder(self::USER_REWARDS_URI);
		$urlBuilder->withPathSite($siteUrl);
		$urlBuilder->withPathUuid($uuid);
		$urlBuilder->withQueryParams($params);

		return $this->httpGet($urlBuilder, $requestOptions);
	}


	/**
	 * Get a user's rewards together with a full list of available rewards.
	 * This returns a list of all the rewards awarded to the specified user
	 * and a list of available rewards. Note: the rewards awarded to the user
	 * will contain a rewardId rather than the reward. The reward can be found
	 * in the list of available rewards.
	 *
	 * @param string $siteUrl					The site to which the user belongs
	 * @param string $uuid						The unique user ID of the user who's reward information is to be retrieved
	 * @param UserGetAllRewardsParams $params	The query parameters to be used with the request
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function getAllRewards(
		string $siteUrl,
		string $uuid,
		UserGetAllRewardsParams $params = null,
		RequestOptions $requestOptions = null
	): array|object {

		$urlBuilder = new ApiUrlBuilder(self::USER_ALL_REWARDS_URI);
		$urlBuilder->withPathSite($siteUrl);
		$urlBuilder->withPathUuid($uuid);
		$urlBuilder->withQueryParams($params);

		return $this->httpGet($urlBuilder, $requestOptions);
	}


	/**
	 * Get a user's achievements. This returns a list of all the achievements
	 * awarded to the specified user.
	 *
	 * @param string $siteUrl					The site to which the user belongs
	 * @param string $uuid						The unique user ID of the user who's achievements information is to be retrieved
	 * @param UserGetAchievementsParams $params	The query parameters to be used with the request
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function getAchievements(
		string $siteUrl,
		string $uuid,
		UserGetAchievementsParams $params = null,
		RequestOptions $requestOptions = null
	): array|object {

		$urlBuilder = new ApiUrlBuilder(self::USER_ACHIEVEMENTS_URI);
		$urlBuilder->withPathSite($siteUrl);
		$urlBuilder->withPathUuid($uuid);
		$urlBuilder->withQueryParams($params);

		return $this->httpGet($urlBuilder, $requestOptions);
	}


	/**
	 * Get a user's achievements together with a full list of available achievements.
	 * This returns a list of all the achievements awarded to the specified user
	 * and a list of available achievements. Note: the achievements awarded to the user
	 * will contain a rewardId rather than the achievement. The achievement can be found
	 * in the list of available achievements.
	 *
	 * @param string $siteUrl						The site to which the user belongs
	 * @param string $uuid							The unique user ID of the user who's achievement information is to be retrieved
	 * @param UserGetAllAchievementsParams $params	The query parameters to be used with the request
	 * @param RequestOptions $requestOptions		The requestOptions to be used with the request
	 */
	public function getAllAchievements(
		string $siteUrl,
		string $uuid,
		UserGetAllAchievementsParams $params = null,
		RequestOptions $requestOptions = null
	): array|object {

		$urlBuilder = new ApiUrlBuilder(self::USER_ALL_ACHIEVEMENTS_URI);
		$urlBuilder->withPathSite($siteUrl);
		$urlBuilder->withPathUuid($uuid);
		$urlBuilder->withQueryParams($params);

		return $this->httpGet($urlBuilder, $requestOptions);
	}


	/**
	 * Get a user's levels. This returns a list of all the levels
	 * awarded to the specified user.
	 *
	 * @param string $siteUrl					The site to which the user belongs
	 * @param string $uuid						The unique user ID of the user who's level information is to be retrieved
	 * @param UserGetLevelsParams $params		The query parameters to be used with the request
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function getLevels(
		string $siteUrl,
		string $uuid,
		UserGetLevelsParams $params = null,
		RequestOptions $requestOptions = null
	): array|object {

		$urlBuilder = new ApiUrlBuilder(self::USER_LEVELS_URI);
		$urlBuilder->withPathSite($siteUrl);
		$urlBuilder->withPathUuid($uuid);
		$urlBuilder->withQueryParams($params);

		return $this->httpGet($urlBuilder, $requestOptions);
	}


	/**
	 * Get a user's levels together with a full list of available levels.
	 * This returns a list of all the levels awarded to the specified user
	 * and a list of available levels. Note: the levels awarded to the user
	 * will contain a rewardId rather than the level. The level can be found
	 * in the list of available levels.
	 *
	 * @param string $siteUrl					The site to which the user belongs
	 * @param string $uuid						The unique user ID of the user who's level information is to be retrieved
	 * @param UserGetAllLevelsParams $params	The query parameters to be used with the request
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function getAllLevels(
		string $siteUrl,
		string $uuid,
		UserGetAllLevelsParams $params = null,
		RequestOptions $requestOptions = null
	): array|object {

		$urlBuilder = new ApiUrlBuilder(self::USER_ALL_LEVELS_URI);
		$urlBuilder->withPathSite($siteUrl);
		$urlBuilder->withPathUuid($uuid);
		$urlBuilder->withQueryParams($params);

		return $this->httpGet($urlBuilder, $requestOptions);
	}


	/**
	 * Get a user's gifts. This returns a list of all the gifts
	 * with the user's progress in them.
	 *
	 * @param string $siteUrl					The site to which the user belongs
	 * @param string $uuid						The unique user ID of the user who's gifts information is to be retrieved
	 * @param UserGetGiftsParams $params		The query parameters to be used with the request
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function getGifts(
		string $siteUrl,
		string $uuid,
		UserGetGiftsParams $params = null,
		RequestOptions $requestOptions = null
	): array|object {

		$urlBuilder = new ApiUrlBuilder(self::USER_GIFTS_URI);
		$urlBuilder->withPathSite($siteUrl);
		$urlBuilder->withPathUuid($uuid);
		$urlBuilder->withQueryParams($params);

		return $this->httpGet($urlBuilder, $requestOptions);
	}


	/**
	 * Get a user's gifts together with a full list of available gifts.
	 * This returns a list of all the gifts awarded to the specified user
	 * and a list of available gifts. Note: the gifts awarded to the user
	 * will contain a rewardId rather than the gift. The gift can be found
	 * in the list of available gifts.
	 *
	 * @param string $siteUrl					The site to which the user belongs
	 * @param string $uuid						The unique user ID of the user who's gift information is to be retrieved
	 * @param UserGetAllGiftsParams $params		The query parameters to be used with the request
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function getAllGifts(
		string $siteUrl,
		string $uuid,
		UserGetAllGiftsParams $params = null,
		RequestOptions $requestOptions = null
	): array|object {

		$urlBuilder = new ApiUrlBuilder(self::USER_ALL_GIFTS_URI);
		$urlBuilder->withPathSite($siteUrl);
		$urlBuilder->withPathUuid($uuid);
		$urlBuilder->withQueryParams($params);

		return $this->httpGet($urlBuilder, $requestOptions);
	}


	/**
	 * Get a user's missions. This returns a list of all the missions
	 * with the user's progress in them.
	 *
	 * @param string $siteUrl					The site to which the user belongs
	 * @param string $uuid						The unique user ID of the user who's missions information is to be retrieved
	 * @param UserGetMissionsParams $params		The query parameters to be used with the request
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function getMissions(
		string $siteUrl,
		string $uuid,
		UserGetMissionsParams $params = null,
		RequestOptions $requestOptions = null
	): array|object {

		$urlBuilder = new ApiUrlBuilder(self::USER_MISSIONS_URI);
		$urlBuilder->withPathSite($siteUrl);
		$urlBuilder->withPathUuid($uuid);
		$urlBuilder->withQueryParams($params);

		return $this->httpGet($urlBuilder, $requestOptions);
	}


	/**
	 * Get a user's missions together with a full list of available missions.
	 * This returns a list of all the missions awarded to the specified user
	 * and a list of available missions. Note: the missions awarded to the user
	 * will contain a rewardId rather than the mission. The mission can be found
	 * in the list of available missions.
	 *
	 * @param string $siteUrl					The site to which the user belongs
	 * @param string $uuid						The unique user ID of the user who's mission information is to be retrieved
	 * @param UserGetAllMissionsParams $params	The query parameters to be used with the request
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function getAllMissions(
		string $siteUrl,
		string $uuid,
		UserGetAllMissionsParams $params = null,
		RequestOptions $requestOptions = null
	): array|object {

		$urlBuilder = new ApiUrlBuilder(self::USER_ALL_MISSIONS_URI);
		$urlBuilder->withPathSite($siteUrl);
		$urlBuilder->withPathUuid($uuid);
		$urlBuilder->withQueryParams($params);

		return $this->httpGet($urlBuilder, $requestOptions);
	}


	/**
	 * Get a user's coupons. This returns a list of all the coupons
	 * with associated user data. Like when they were awarded or redeemed on.
	 *
	 * @param string $siteUrl					The site to which the user belongs
	 * @param string $uuid						The unique user ID of the user who's coupons information is to be retrieved
	 * @param UserGetCouponsParams $params		The query parameters to be used with the request
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function getCoupons(
		string $siteUrl,
		string $uuid,
		UserGetCouponsParams $params = null,
		RequestOptions $requestOptions = null
	): array|object {

		$urlBuilder = new ApiUrlBuilder(self::USER_COUPONS_URI);
		$urlBuilder->withPathSite($siteUrl);
		$urlBuilder->withPathUuid($uuid);
		$urlBuilder->withQueryParams($params);

		return $this->httpGet($urlBuilder, $requestOptions);
	}


	/**
	 * Get a user's coupons and a list of coupons which can be bought with points.
	 *
	 * @param string $siteUrl					The site to which the user belongs
	 * @param string $uuid						The unique user ID of the user who's coupons information is to be retrieved
	 * @param UserGetAllCouponsParams $params	The query parameters to be used with the request
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function getAllCoupons(
		string $siteUrl,
		string $uuid,
		UserGetAllCouponsParams $params = null,
		RequestOptions $requestOptions = null
	): array|object {

		$urlBuilder = new ApiUrlBuilder(self::USER_ALL_COUPONS_URI);
		$urlBuilder->withPathSite($siteUrl);
		$urlBuilder->withPathUuid($uuid);
		$urlBuilder->withQueryParams($params);

		return $this->httpGet($urlBuilder, $requestOptions);
	}


	/**
	 * Get the leaderboard specified by ID  with the user specified.
	 * The regular behaviour is to return the Top 10 users with a contextUser that
	 * contains the user specified. If the contextual flag is set, then the contextUser
	 * will be returned at the center of the leaderboard users, with 10 users above
	 * and 10 users below inluded.
	 *
	 * @param string $siteUrl					The site for which to retrieve the system leaderboard
	 * @param string $uuid						The unique user ID of the user who's leaderboard information is to be retrieved
	 * @param string $id						The ID of the leaderboard to retrieve
	 * @param UserGetLeaderboardParams $params	The query parameters to be used with the request
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function getLeaderboard(
		string $siteUrl,
		string $uuid,
		string $id,
		UserGetLeaderboardParams $params = null,
		RequestOptions $requestOptions = null
	): array|object {

		$urlBuilder = new ApiUrlBuilder(self::USER_LEADERBOARD_URI);
		$urlBuilder->withPathSite($siteUrl);
		$urlBuilder->withPathUuid($uuid);
		$urlBuilder->withQueryParams($params);

		return $this->httpGet($urlBuilder, $requestOptions);
	}


	/**
	 * Get a user's notifications
	 *
	 * @param string $siteUrl						The site to which the user belongs
	 * @param string $uuid							The unique user ID of the user who's notifications information is to be retrieved
	 * @param UserGetNotificationsParams $params	The query parameters to be used with the request
	 * @param RequestOptions $requestOptions		The requestOptions to be used with the request
	 */
	public function getNotifications(
		string $siteUrl,
		string $uuid,
		UserGetNotificationsParams $params = null,
		RequestOptions $requestOptions = null
	): array|object {

		$urlBuilder = new ApiUrlBuilder(self::USER_NOTIFICATIONS_URI);
		$urlBuilder->withPathSite($siteUrl);
		$urlBuilder->withPathUuid($uuid);
		$urlBuilder->withQueryParams($params);

		return $this->httpGet($urlBuilder, $requestOptions);
	}


	/**
	 * Clear a user's notifications
	 *
	 * @param string $siteUrl	The site to which the user belongs
	 * @param string $uuid		The unique user ID of the user who's notifications are to be cleared
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function clearNotifications(string $siteUrl, string $uuid, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::USER_CLEAR_NOTIFICATIONS_URI);
		$urlBuilder->withPathSite($siteUrl);
		$urlBuilder->withPathUuid($uuid);
		return $this->httpPost($urlBuilder, $requestOptions);
	}


	/**
	 * Get a user's activities. This returns a paged list of the activities
	 * performed by the user.
	 *
	 * @param string $siteUrl	The site to which the user belongs
	 * @param string $uuid		The unique user ID of the user who's activities information is to be retrieved
	 * @param UserGetActivitiesParams $params	The query parameters to be used with the request
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function getActivities(
		string $siteUrl,
		string $uuid,
		UserGetActivitiesParams $params = null,
		RequestOptions $requestOptions = null
	): array|object {

		$urlBuilder = new ApiUrlBuilder(self::USER_ACTIVITIES_URI);
		$urlBuilder->withPathSite($siteUrl);
		$urlBuilder->withPathUuid($uuid);
		$urlBuilder->withQueryParams($params);

		return $this->httpGet($urlBuilder, $requestOptions);
	}


	/**
	 * Get a user's KPIs. This returns a paged list of the KPIs
	 * assigned to the user.
	 *
	 * @param string $siteUrl					The site to which the user belongs
	 * @param string $uuid						The unique user ID of the user who's activities information is to be retrieved
	 * @param UserGetKpisParams $params			The query parameters to be used with the request
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function getKpis(
		string $siteUrl,
		string $uuid,
		UserGetKpisParams $params = null,
		RequestOptions $requestOptions = null
	): array|object {

		$urlBuilder = new ApiUrlBuilder(self::USER_KPIS_URI);
		$urlBuilder->withPathSite($siteUrl);
		$urlBuilder->withPathUuid($uuid);
		$urlBuilder->withQueryParams($params);

		return $this->httpGet($urlBuilder, $requestOptions);
	}


	/**
	 * Get a user's point wallets. This returns a paged list of the wallets assigned to the user
	 *
	 * @param string $siteUrl					The site to which the user belongs
	 * @param string $uuid						The unique user ID of the user who's wallets information is to be retrieved
	 * @param UserGetPointWalletsParams $params	The query parameters to be used with the request
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function getPointWallets(
		string $siteUrl,
		string $uuid,
		UserGetPointWalletsParams $params = null,
		RequestOptions $requestOptions = null
	): array|object {

		$urlBuilder = new ApiUrlBuilder(self::USER_POINT_WALLETS_URI);
		$urlBuilder->withPathSite($siteUrl);
		$urlBuilder->withPathUuid($uuid);
		$urlBuilder->withQueryParams($params);

		return $this->httpGet($urlBuilder, $requestOptions);
	}


	/**
	 * Get a user's point transactions. This returns a paged list of the transaction assigned to the user
	 *
	 * @param string $siteUrl							The site to which the user belongs
	 * @param string $uuid								The unique user ID of the user who's transaction information is to be retrieved
	 * @param string $pointId							The ID of the point that we need the transactions for
	 * @param UserGetPointTransactionsParams $params	The query parameters to be used with the request
	 * @param RequestOptions $requestOptions			The requestOptions to be used with the request
	 */
	public function getPointTransactions(
		string $siteUrl,
		string $uuid,
		string $pointId,
		UserGetPointTransactionsParams $params = null,
		RequestOptions $requestOptions = null
	): array|object {

		$urlBuilder = new ApiUrlBuilder(self::USER_POINT_TRANSACTIONS_URI);
		$urlBuilder->withPathSite($siteUrl);
		$urlBuilder->withPathUuid($uuid);
		$urlBuilder->withPathParam("pointId", $pointId);
		$urlBuilder->withQueryParams($params);

		return $this->httpGet($urlBuilder, $requestOptions);
	}


	/**
	 * This method is used to add custom fields to an existing user
	 *
	 * @param string $id						The ID of the user to update
	 * @param CustomFieldValueRequestData $data	The list of custom fields to add
	 * @param UserCustomFieldParams $params		The query parameters to be used with the request
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function addCustomFields(
		string $id,
		CustomFieldValueRequestData $data,
		UserCustomFieldParams $params,
		RequestOptions $requestOptions = null
	): array|object {
		return $this->modCustomFields(HttpClient::POST, $id, $data, $params, $requestOptions);
	}


	/**
	 * This method is used to update custom fields in an existing user
	 *
	 * @param string $id						The ID of the user to update
	 * @param CustomFieldValueRequestData $data	The list of custom fields to update
	 * @param UserCustomFieldParams $params		The query parameters to be used with the request
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function updateCustomFields(
		string $id,
		CustomFieldValueRequestData $data,
		UserCustomFieldParams $params,
		RequestOptions $requestOptions = null
	): array|object {
		return $this->modCustomFields(HttpClient::PUT, $id, $data, $params, $requestOptions);
	}


	/**
	 * This method is used to delete custom fields in an existing user
	 *
	 * @param string $id						The ID of the user to update
	 * @param CustomFieldValueRequestData $data	The list of custom fields to update
	 * @param UserCustomFieldParams $params		The query parameters to be used with the request
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function deleteCustomFields(
		string $id,
		CustomFieldValueRequestData $data,
		UserCustomFieldParams $params,
		RequestOptions $requestOptions = null
	): array|object {
		return $this->modCustomFields(HttpClient::DELETE, $id, $data, $params, $requestOptions);
	}


	/**
	 * Modifying custom fields
	 */
	private function modCustomFields(
		string $method,
		string $id,
		CustomFieldValueRequestData $data,
		UserCustomFieldParams $params,
		RequestOptions $requestOptions = null
	): array|object {

		$urlBuilder = new ApiUrlBuilder(self::USER_CUSTOM_URI);
		$urlBuilder->withPathId($id);
		$urlBuilder->withQueryParams($params);

		return $this->client->request($urlBuilder, $method, $data, $requestOptions);
	}


	/**
	 * This method is used to add a custom field with an image to an existing user
	 *
	 * @param string $id						The ID of the user to update
	 * @param CustomFieldValue $data	        The custom field to update
	 * @param UserCustomFieldParams $params		The query parameters to be used with the request
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function addCustomField(
		string $id,
		CustomFieldValue $data,
		string|false $image,
		UserCustomFieldParams $params = null,
		RequestOptions $requestOptions = null
	): array|object {

		Preconditions::checkNotEmpty($image, "image must not be empty");

		$urlBuilder = new ApiUrlBuilder(self::USER_CUSTOM_IMAGE_URI);
		$urlBuilder->withPathId($id);
		$urlBuilder->withQueryParams($params);

		return $this->httpUpload($urlBuilder, $image, $requestOptions, $data);
	}
}
