<?php
/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

declare(strict_types=1);

namespace Mambo\User\Param;

use Mambo\Common\Param\AbstractServiceParams;
use Mambo\Common\Param\Order;

/**
 * Parameters for searching users
 * @extends AbstractServiceParams<UserSearchParams>
 */
class UserSearchParams extends AbstractServiceParams
{
    /**
     * Specifies the page of results to retrieve.
     * 
     * @param int $page
     * @return self
     */
    public function page(int $page): self
    {
        return $this->extraParams('page', $page);
    }

    /**
     * Specifies the number of results to retrieve, up to a maximum of 100.  
     *
     * @param int $count
     * @return self
     */
    public function count(int $count): self
    {
        return $this->extraParams('count', $count);
    }

    /**
     * The list of tags to filter by
     * 
     * @param array<string> $tags
     * @return self
     */
    public function tags(array $tags): self
    {
        return $this->extraParamsList('tags', $tags);
    }

    /**
     * Whether tags should filter for any or all tags
     * 
     * @param string $tagsJoin
     * @return self
     */
    public function tagsJoin(string $tagsJoin): self
    {
        return $this->extraParams('tagsJoin', $tagsJoin);
    }

    /**
     * Field to order results by
     * 
     * @param string $orderBy
     * @return self
     */
    public function orderBy(string $orderBy): self
    {
        return $this->extraParams('orderBy', $orderBy);
    }

    /**
     * Sort order for results
     * 
     * @param string $order
     * @return self
     */
    public function order(string $order): self
    {
        return $this->extraParams('order', $order);
    }

    /**
     * Whether to include the users stats
     * 
     * @param bool $withStats
     * @return self
     */
    public function withStats(bool $withStats): self
    {
        return $this->extraParams('withStats', $withStats);
    }

    /**
     * User properties to return
     * 
     * @param array<string> $properties
     * @return self
     */
    public function properties(array $properties): self
    {
        return $this->extraParamsList('props', $properties);
    }

    /**
     * Reward IDs to filter by
     * 
     * @param array<string> $rewardIds
     * @return self
     */
    public function rewardIds(array $rewardIds): self
    {
        return $this->extraParamsList('rewardIds', $rewardIds);
    }

    /**
     * Behaviour IDs to filter by
     * 
     * @param array<string> $behaviourIds
     * @return self
     */
    public function behaviourIds(array $behaviourIds): self
    {
        return $this->extraParamsList('behaviourIds', $behaviourIds);
    }

    /**
     * Users last seen on starting from
     * 
     * @param string $lastSeenOnStart
     * @return self
     */
    public function lastSeenOnStart(string $lastSeenOnStart): self
    {
        return $this->extraParams('lastSeenOnStart', $lastSeenOnStart);
    }

    /**
     * Users last seen on ending on
     * 
     * @param string $lastSeenOnEnd
     * @return self
     */
    public function lastSeenOnEnd(string $lastSeenOnEnd): self
    {
        return $this->extraParams('lastSeenOnEnd', $lastSeenOnEnd);
    }

    /**
     * Search query
     * 
     * @param string $searchQuery
     * @return self
     */
    public function searchQuery(string $searchQuery): self
    {
        return $this->extraParams('query', $searchQuery);
    }

    /**
     * Fields to search on
     * 
     * @param array<string> $searchFields
     * @return self
     */
    public function searchFields(array $searchFields): self
    {
        return $this->extraParamsList('searchFields', $searchFields);
    }
}
