<?php
/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

declare(strict_types=1);

namespace Mambo\User\Param;

use Mambo\Common\Param\AbstractServiceParams;

/**
 * Parameters for getting user activities
 * @extends AbstractServiceParams<UserGetActivitiesParams>
 */
class UserGetActivitiesParams extends AbstractServiceParams
{
    /**
     * Specifies the page of results to retrieve.
     */
    public function page(int $page): self
    {
        return $this->extraParams('page', $page);
    }

    /**
     * Specifies the number of results to retrieve, up to a maximum of 100.
     */
    public function count(int $count): self
    {
        return $this->extraParams('count', $count);
    }

    /**
     * The list of tags to filter by
     * @param array<string> $tags
     */
    public function tags(array $tags): self
    {
        return $this->extraParamsList('tags', $tags);
    }

    /**
     * Whether tags should filter for any or all tags
     */
    public function tagsJoin(string $tagsJoin): self
    {
        return $this->extraParams('tagsJoin', $tagsJoin);
    }

    /**
     * Whether to filter by personalization tags
     */
    public function withPersonalization(bool $withPersonalization): self
    {
        return $this->extraParams('withPersonalization', $withPersonalization);
    }

    /**
     * Specifies whether only activities which unlocked one or more behaviours should be returned.
     */
    public function withBehavioursOnly(bool $withBehavioursOnly): self
    {
        return $this->extraParams('withBehavioursOnly', $withBehavioursOnly);
    }

    /**
     * Specifies whether only activities which unlocked one or more rewards should be returned.
     */
    public function withRewardsOnly(bool $withRewardsOnly): self
    {
        return $this->extraParams('withRewardsOnly', $withRewardsOnly);
    }

    /**
     * Specifies whether only activities which unlocked one or more missions should be returned.
     */
    public function withMissionsOnly(bool $withMissionsOnly): self
    {
        return $this->extraParams('withMissionsOnly', $withMissionsOnly);
    }

    /**
     * Specifies whether only activities which unlocked one or more levels should be returned.
     */
    public function withLevelsOnly(bool $withLevelsOnly): self
    {
        return $this->extraParams('withLevelsOnly', $withLevelsOnly);
    }

    /**
     * Specifies whether only activities which unlocked one or more achievements should be returned.
     */
    public function withAchievementsOnly(bool $withAchievementsOnly): self
    {
        return $this->extraParams('withAchievementsOnly', $withAchievementsOnly);
    }

    /**
     * Specifies whether only activities which unlocked one or more gifts should be returned.
     */
    public function withGiftsOnly(bool $withGiftsOnly): self
    {
        return $this->extraParams('withGiftsOnly', $withGiftsOnly);
    }

    /**
     * Specifies whether only activities which changed one or more KPIs should be returned.
     */
    public function withKPIsOnly(bool $withKPIsOnly): self
    {
        return $this->extraParams('withKPIsOnly', $withKPIsOnly);
    }

    /**
     * Specifies whether only activities which were compensated should be returned
     */
    public function withCompensatedOnly(bool $withCompensatedOnly): self
    {
        return $this->extraParams('withCompensatedOnly', $withCompensatedOnly);
    }

    /**
     * Specifies whether only activities which have been rejected should be returned.
     */
    public function withRejectedOnly(bool $withRejectedOnly): self
    {
        return $this->extraParams('withRejectedOnly', $withRejectedOnly);
    }

    /**
     * Specifies whether the behaviour activities returned should include behaviours which have an exception status.
     */
    public function withExceptions(bool $withExceptions): self
    {
        return $this->extraParams('withExceptions', $withExceptions);
    }

    /**
     * Include activities with matching target user
     */
    public function withTargetUser(bool $withTargetUser): self
    {
        return $this->extraParams('withTargetUser', $withTargetUser);
    }

    /**
     * Specifies whether the response should have nested objects populated.
     */
    public function populated(bool $populated): self
    {
        return $this->extraParams('populated', $populated);
    }

    /**
     * Sets the start date for filtering
     */
    public function startDate(string $startDate): self
    {
        return $this->extraParams('startDate', $startDate);
    }

    /**
     * Sets the end date for filtering
     */
    public function endDate(string $endDate): self
    {
        return $this->extraParams('endDate', $endDate);
    }

    /**
     * The list of reward IDs to filter by (if any)
     * @param array<string> $rewardIds
     */
    public function rewardIds(array $rewardIds): self
    {
        return $this->extraParamsList('rewardIds', $rewardIds);
    }

    /**
     * The list of behaviour IDs to filter by (if any)
     * @param array<string> $behaviourIds
     */
    public function behaviourIds(array $behaviourIds): self
    {
        return $this->extraParamsList('behaviourIds', $behaviourIds);
    }

    /**
     * The list of KPI IDs to filter by (if any)
     * @param array<string> $kpiIds
     */
    public function kpiIds(array $kpiIds): self
    {
        return $this->extraParamsList('kpiIds', $kpiIds);
    }

    /**
     * The list of point IDs to filter by (if any)
     * @param array<string> $pointIds
     * @deprecated
     */
    public function pointIds(array $pointIds): self
    {
        return $this->extraParamsList('pointIds', $pointIds);
    }
}
