<?php
/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * The MamboTagsService class handles all Tag related requests
 * to the Mambo API.
 */

declare(strict_types=1);

namespace Mambo\Tag;

use Mambo\Tag\Data\TagRequestData;
use Mambo\Common\Service\AbstractService;
use Mambo\Common\Data\DeleteRequestData;
use Mambo\Common\Data\CustomFieldValueRequestData;
use Mambo\Common\Data\CustomFieldValue;
use Mambo\Http\ApiUrlBuilder;
use Mambo\Http\RequestOptions;
use Mambo\Http\HttpClient;
use Mambo\Util\Preconditions;

class TagsService extends AbstractService
{
	/**
	 * Tag service end tag URI
	 * @var string
	 */
	const TAGS_URI = "/v1/tags";
	const TAGS_ID_URI = "/v1/tags/{id}";
	const TAGS_CLONE_URI = "/v1/tags/{id}/clone";
	const TAGS_CUSTOM_URI = "/v1/tags/{id}/custom_fields";
	const TAGS_CUSTOM_IMAGE_URI = "/v1/tags/{id}/custom_fields/image";

	const TAGS_SITE_URI = "/v1/{site}/tags";
	const TAGS_TAGS_SITE_URI = "/v1/{site}/tags/{tagTag}";


	/**
	 * This method is used to create a new tag.
	 *
	 * @param string $siteUrl					The site to which the tag belongs to
	 * @param TagRequestData $data				The data sent to the API in order to create a tag
	 * @param RequestOptions $requestOptions		The requestOptions to be used with the request
	 */
	public function create(string $siteUrl, TagRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::TAGS_SITE_URI);
		$urlBuilder->withPathSite($siteUrl);
		return $this->httpPost($urlBuilder, $requestOptions, $data);
	}


	/**
	 * Update an existing tag by ID.
	 *
	 * @param string $id					The ID of the tag to update
	 * @param TagRequestData $data	The data with which to update the specified tag object
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function update(string $id, TagRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::TAGS_ID_URI);
		$urlBuilder->withPathId($id);
		return $this->httpPut($urlBuilder, $requestOptions, $data);
	}


	/**
	 * Clone a tag
	 *
	 * @param string $id		The ID of the tag to clone
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function cloneTag(string $id, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::TAGS_CLONE_URI);
		$urlBuilder->withPathId($id);
		return $this->httpPost($urlBuilder, $requestOptions);
	}


	/**
	 * Delete a tag by it's ID
	 *
	 * @param string $id				The ID of the tag to delete
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function delete(string $id, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::TAGS_ID_URI);
		$urlBuilder->withPathId($id);
		return $this->httpDelete($urlBuilder, $requestOptions);
	}


	/**
	 * Delete a list of tags by their ID
	 *
	 * @param DeleteRequestData $data		The list of IDs of the tag to delete
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function deleteTags(DeleteRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::TAGS_URI);
		return $this->httpDelete($urlBuilder, $requestOptions, $data);
	}


	/**
	 * Get a tag by it's site and tag value.
	 *
	 * @param string $siteUrl	The site to which the tag belongs to
	 * @param string $tagTag		The tag value of the Tag we want to retrieve
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function getByTagTag(string $siteUrl, string $tagTag, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::TAGS_TAGS_SITE_URI);
		$urlBuilder->withPathSite($siteUrl);
		$urlBuilder->withPathParam("tagTag", $tagTag);
		return $this->httpGet($urlBuilder, $requestOptions);
	}


	/**
	 * Get a tag by it's ID
	 *
	 * @param string $id			The ID of the tag to retrieve
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function get(string $id, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::TAGS_ID_URI);
		$urlBuilder->withPathId($id);
		return $this->httpGet($urlBuilder, $requestOptions);
	}


	/**
	 * Get the list of tags for the specified site
	 *
	 * @param string $siteUrl	The site for which to retrieve the list of tags
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function getTags(string $siteUrl, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::TAGS_SITE_URI);
		$urlBuilder->withPathSite($siteUrl);
		return $this->httpGet($urlBuilder, $requestOptions);
	}


	/**
	 * This method is used to add custom fields to an existing tag
	 *
	 * @param string $id						The ID of the tag to update
	 * @param CustomFieldValueRequestData $data	The list of custom fields to add
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function addCustomFields(string $id, CustomFieldValueRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		return $this->modCustomFields(HttpClient::POST, $id, $data, $requestOptions);
	}


	/**
	 * This method is used to update custom fields in an existing tag
	 *
	 * @param string $id						The ID of the tag to update
	 * @param CustomFieldValueRequestData $data	The list of custom fields to update
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function updateCustomFields(string $id, CustomFieldValueRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		return $this->modCustomFields(HttpClient::PUT, $id, $data, $requestOptions);
	}


	/**
	 * This method is used to delete custom fields in an existing tag
	 *
	 * @param string $id						The ID of the tag to update
	 * @param CustomFieldValueRequestData $data	The list of custom fields to update
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function deleteCustomFields(string $id, CustomFieldValueRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		return $this->modCustomFields(HttpClient::DELETE, $id, $data, $requestOptions);
	}


	/**
	 * Modifying custom fields
	 */
	private function modCustomFields(string $method, string $id, CustomFieldValueRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::TAGS_CUSTOM_URI);
		$urlBuilder->withPathId($id);
		return $this->client->request($urlBuilder, $method, $data, $requestOptions);
	}


	/**
	 * This method is used to add a custom field with an image in an existing tag
	 *
	 * @param string $id						The ID of the tag to update
	 * @param CustomFieldValue $data         The custom field to update
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function addCustomField(string $id, CustomFieldValue $data, string|false $image, RequestOptions $requestOptions = null): array|object
	{
		Preconditions::checkNotEmpty($image, "The image must not be empty");

		$urlBuilder = new ApiUrlBuilder(self::TAGS_CUSTOM_IMAGE_URI);
		$urlBuilder->withPathId($id);
		return $this->httpUpload($urlBuilder, $image, $requestOptions, $data);
	}
}
