<?php
/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * The MamboSitesService class handles all Site related requests
 * to the Mambo API.
 */

declare(strict_types=1);

namespace Mambo\Site;

use Mambo\Site\Data\SiteRequestData;
use Mambo\Common\Service\AbstractService;
use Mambo\Common\Data\DeleteRequestData;
use Mambo\Http\ApiUrlBuilder;
use Mambo\Http\RequestOptions;

class SitesService extends AbstractService
{
	/**
	 * Site service end point URI
	 * @var string
	 */
	const SITES_URI = "/v1/sites";
	const SITES_ID_URI = "/v1/sites/{id}";
	const SITES_CLONE_URI = "/v1/sites/{id}/clone";


	/**
	 * This method is used to create a new site.
	 *
	 * @param SiteRequestData $data				The data sent to the API in order to create a site
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function create(SiteRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::SITES_URI);
		return $this->httpPost($urlBuilder, $requestOptions, $data);
	}


	/**
	 * Update an existing site by ID.
	 *
	 * @param string $id				The ID of the site to update
	 * @param SiteRequestData $data		The data with which to update the specified site object
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function update(string $id, SiteRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::SITES_ID_URI);
		$urlBuilder->withPathId($id);
		return $this->httpPut($urlBuilder, $requestOptions, $data);
	}


	/**
	 * Clone an existing site by ID.
	 *
	 * @param string $id				The ID of the site to clone
	 * @param SiteRequestData $data 	The data of the site into which we wish to clone
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function cloneSite(string $id, SiteRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::SITES_CLONE_URI);
		$urlBuilder->withPathId($id);
		return $this->httpPost($urlBuilder, $requestOptions, $data);
	}


	/**
	 * Delete a site by it's ID
	 *
	 * @param string $id				The ID of the site to delete
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function delete(string $id, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::SITES_ID_URI);
		$urlBuilder->withPathId($id);
		return $this->httpDelete($urlBuilder, $requestOptions);
	}


	/**
	 * Delete a list of sites by their ID
	 *
	 * @param DeleteRequestData $data		The list of IDs of the behaviour to delete
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function deleteSites(DeleteRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::SITES_URI);
		return $this->httpDelete($urlBuilder, $requestOptions, $data);
	}


	/**
	 * Get a site by it's ID
	 *
	 * @param string $id			The ID of the site to retrieve
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function get(string $id, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::SITES_ID_URI);
		$urlBuilder->withPathId($id);
		return $this->httpGet($urlBuilder, $requestOptions);
	}


	/**
	 * Get the list of sites available
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function getSites(RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::SITES_URI);
		return $this->httpGet($urlBuilder, $requestOptions);
	}
}
