<?php
/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * This object captures the data required by the Site API in
 * order to create / update sites.
 */

declare(strict_types=1);

namespace Mambo\Site\Data;

use Mambo\Common\Data\RequestData;
use Mambo\Site\Data\Authentication\AuthenticationContextDto;
use Mambo\Site\Data\Authentication\AuthenticationRedirectDto;

class SiteRequestData implements RequestData
{
	private $data = array();

	/**
	 * The name by which this site network is known.
	 * @return string name
	 */
	public function getName()
	{
		return $this->data['name'];
	}
	public function setName($name)
	{
		$this->data['name'] = $name;
	}


	/**
	 * The URL for this specific site network. URLs do not
	 * have to be legitimate domain names. The URLs should
	 * be easily understandable. For example:
	 * development.acme.com
	 * staging.acme.com
	 * live.acme.com
	 * @return string url
	 */
	public function getUrl()
	{
		return $this->data['url'];
	}
	public function setUrl($url)
	{
		$this->data['url'] = $url;
	}


	/**
	 * The time zone to be used for this site. The time zone
	 * is used to determine when to run certain Mambo operations.
	 * The operations include:
	 * - Changing time periods in the Leaderboards
	 * - Changing time periods in KPIs
	 * - Aggregating analytics data job
	 * Time zones must be supplied using their Olson name.
	 * For example: Australia/Darwin, Europe/Paris, Pacific/Port_Moresby
	 * @see https://www.iana.org/time-zones
	 */
	public function getTimezone()
	{
		return $this->data['timezone'];
	}
	public function setTimezone($timezone)
	{
		$this->data['timezone'] = $timezone;
	}


	/**
	 * The first day of the week to be used for this site. For example, your week
	 * may start on Sunday or on Monday. This affects how data is stored by week.
	 * Valid range of values: 1 - 7
	 * Where 1 is Monday and 7 is Sunday.
	 */
	public function getFirstDayOfWeek()
	{
		return $this->data['firstDayOfWeek'];
	}
	public function setFirstDayOfWeek($firstDayOfWeek)
	{
		$this->data['firstDayOfWeek'] = $firstDayOfWeek;
	}


	/**
	 * The Security object is used to define the security settings for the site.
	 * @return Security security
	 */
	public function getSecurity()
	{
		return $this->data['security'];
	}
	public function setSecurity(Security $security)
	{
		$this->data['security'] = $security;
	}


	/**
	 * The redirection endpoints for the site after authentication.
	 * @return AuthenticationRedirectDto redirect
	 */
	public function getRedirect()
	{
		return $this->data['redirect'];
	}
	public function setRedirect(AuthenticationRedirectDto $redirect)
	{
		$this->data['redirect'] = $redirect;
	}


	/**
	 * The Authentication method for the site.
	 * @return AuthenticationContextDto authentication
	 */
	public function getAuthentication()
	{
		return $this->data['authentication'];
	}
	public function setAuthentication(AuthenticationContextDto $authentication)
	{
		$this->data['authentication'] = $authentication;
	}


	/**
	 * Return the JSON string equivalent of this object
	 */
	public function getJsonString(): string
	{
		$json = $this->data;

		if (isset($json['security'])) {
			$json['security'] = $json['security']->getJsonArray();
		}

		if (isset($json['redirect'])) {
			$json['redirect'] = $json['redirect']->getJsonArray();
		}

		if (isset($json['authentication'])) {
			$json['authentication'] = $json['authentication']->getJsonArray();
		}

		return json_encode($json);
	}
}
