<?php
/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * This class specify the context with which the LDAP authentication runs.
 */

declare(strict_types=1);

namespace Mambo\Site\Data\Authentication;

class LdapAuthenticationContextDto implements AuthenticationContextDto
{
    private $data = array();

    /**
     * Hostname of the server running LDAP. Example: ldap.domain.com
     */
    public function getHostname()
    {
        return $this->data['hostname'];
    }
    public function setHostname($hostname)
    {
        $this->data['hostname'] = $hostname;
    }

    /**
     * The port number of the LDAP server to connect with.
     */
    public function getPort()
    {
        return $this->data['port'];
    }
    public function setPort($port)
    {
        $this->data['port'] = $port;
    }

    /**
     * Whether we should use SSL to connect to LDAP
     */
    public function getUseSsl()
    {
        return $this->data['useSsl'];
    }
    public function setUseSsl($useSsl)
    {
        $this->data['useSsl'] = $useSsl;
    }

    /**
     * Root node in LDAP from which to search for users.
     * Example: cn=users,dc=domain,dc=com
     */
    public function getBaseDn()
    {
        return $this->data['baseDn'];
    }
    public function setBaseDn($baseDn)
    {
        $this->data['baseDn'] = $baseDn;
    }

    /**
     * The user distinguished name to use for getting
     * authenticated contexts.
     *
     */
    public function getUserDn()
    {
        return $this->data['userDn'];
    }
    public function setUserDn($userDn)
    {
        $this->data['userDn'] = $userDn;
    }

    /**
     * The password to connect with the LDAP authentication server.
     *
     */
    public function getPassword()
    {
        return $this->data['password'];
    }
    public function setPassword($password)
    {
        $this->data['password'] = $password;
    }

    /**
     * User search filter uses RFC 2254 to query LDAP for the
     * user which should be used during login. The filter should
     * contain {0} which will be substituted for the username
     * during the login process. Ensure the search filter returns
     * a single user otherwise the login will return an error.
     */
    public function getUserSearchFilter()
    {
        return $this->data['userSearchFilter'];
    }
    public function setUserSearchFilter($userSearchFilter)
    {
        $this->data['userSearchFilter'] = $userSearchFilter;
    }

    /**
     * The UUID of a user to test the userSearchFilter.
     * Ex: admin
     */
    public function getUserSearchFilterTestUser()
    {
        return $this->data['userSearchFilterTestUser'];
    }
    public function setUserSearchFilterTestUser($userSearchFilterTestUser)
    {
        $this->data['userSearchFilterTestUser'] = $userSearchFilterTestUser;
    }

    /**
     * Contains the mapping information between an external
     * user directory and the internal Mambo users.
     */
    public function getSyncUserAttributes()
    {
        return $this->data['syncUserAttributes'];
    }
    public function setSyncUserAttributes($syncUserAttributes)
    {
        $this->data['syncUserAttributes'] = $syncUserAttributes;
    }

    /**
     * The type of AuthenticationRequestData.
     *
     */
    public function getAuthenticationType()
    {
        return AuthenticationType::LDAP;
    }

    /**
     * The type of authentication context parameter settings attribute.
     *
     */
    public function getType()
    {
        return "ldap";
    }

    /**
     * Returns the current model as an array ready to
     * be json_encoded
     */
    public function getJsonArray()
    {
        $json = $this->data;
        $json['type'] = $this->getType();
        return $json;
    }
}
