<?php
/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * The MamboRewardsService class handles all Reward related requests
 * to the Mambo API.
 */

declare(strict_types=1);

namespace Mambo\Reward;

use Mambo\Reward\Data\RewardRequestData;
use Mambo\Reward\Param\RewardGetAchievementListParams;
use Mambo\Reward\Param\RewardGetListParams;
use Mambo\Reward\Param\RewardGetGiftListParams;
use Mambo\Reward\Param\RewardGetLevelListParams;
use Mambo\Reward\Param\RewardGetMissionListParams;
use Mambo\Common\Service\AbstractService;
use Mambo\Common\Data\DeleteRequestData;
use Mambo\Common\Data\CustomFieldValueRequestData;
use Mambo\Common\Data\CustomFieldValue;
use Mambo\Http\ApiUrlBuilder;
use Mambo\Http\RequestOptions;
use Mambo\Http\HttpClient;
use Mambo\Util\Preconditions;

class RewardsService extends AbstractService
{
	/**
	 * Reward service end point URIs
	 * @var string
	 */
	const REWARDS_URI = "/v1/rewards";
	const REWARDS_ID_URI = "/v1/rewards/{id}";
	const REWARDS_IMAGE_URI = "/v1/rewards/{id}/image";
	const REWARDS_CLONE_URI = "/v1/rewards/{id}/clone";
	const REWARDS_CUSTOM_URI = "/v1/rewards/{id}/custom_fields";
	const REWARDS_CUSTOM_IMAGE_URI = "/v1/rewards/{id}/custom_fields/image";

	const REWARDS_SITE_URI = "/v1/{site}/rewards";
	const ACHIEVEMENTS_SITE_URI = "/v1/{site}/rewards/achievements";
	const LEVELS_SITE_URI = "/v1/{site}/rewards/levels";
	const MISSIONS_SITE_URI = "/v1/{site}/rewards/missions";
	const GIFTS_SITE_URI = "/v1/{site}/rewards/gifts";


	/**
	 * This method is used to create a new reward.
	 *
	 * @param string $siteUrl					The site to which the reward belongs to
	 * @param RewardRequestData $data			The data sent to the API in order to create a reward
	 * @param RequestOptions $requestOptions		The requestOptions to be used with the request
	 */
	public function create(string $siteUrl, RewardRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::REWARDS_SITE_URI);
		$urlBuilder->withPathSite($siteUrl);
		return $this->httpPost($urlBuilder, $requestOptions, $data);
	}


	/**
	 * Update an existing reward by ID.
	 *
	 * @param string $id					The ID of the reward to update
	 * @param RewardRequestData $data	The data with which to update the specified reward object
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function update(string $id, RewardRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::REWARDS_ID_URI);
		$urlBuilder->withPathId($id);
		return $this->httpPut($urlBuilder, $requestOptions, $data);
	}


	/**
	 * Upload an image for the reward
	 *
	 * @param string $id		The reward for which to upload the image
	 * @param string|false $image 	The image to upload for the reward
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function uploadImage(string $id, string|false $image, RequestOptions $requestOptions = null): array|object
	{
		Preconditions::checkNotEmpty($image, "The image must not be empty");

		$urlBuilder = new ApiUrlBuilder(self::REWARDS_IMAGE_URI);
		$urlBuilder->withPathId($id);
		return $this->httpUpload($urlBuilder, $image, $requestOptions);
	}


	/**
	 * Clone a reward
	 *
	 * @param string $id		The ID of the reward to clone
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function cloneReward(string $id, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::REWARDS_CLONE_URI);
		$urlBuilder->withPathId($id);
		return $this->httpPost($urlBuilder, $requestOptions);
	}


	/**
	 * Delete a reward by it's ID
	 *
	 * @param string $id				The ID of the reward to delete
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function delete(string $id, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::REWARDS_ID_URI);
		$urlBuilder->withPathId($id);
		return $this->httpDelete($urlBuilder, $requestOptions);
	}


	/**
	 * Delete a list of rewards by their ID
	 *
	 * @param DeleteRequestData $data		The list of IDs of the reward to delete
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function deleteRewards(DeleteRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::REWARDS_URI);
		return $this->httpDelete($urlBuilder, $requestOptions, $data);
	}


	/**
	 * Get a reward by it's ID
	 *
	 * @param string $id			The ID of the reward to retrieve
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function get(string $id, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::REWARDS_ID_URI);
		$urlBuilder->withPathId($id);
		return $this->httpGet($urlBuilder, $requestOptions);
	}


	/**
	 * Get the list of rewards for the specified site
	 *
	 * @param string $siteUrl					The site for which to retrieve the list of rewards
	 * @param RewardGetListParams $params		The query parameters to be used with the request
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function getRewards(
		string $siteUrl,
		RewardGetListParams $params = null,
		RequestOptions $requestOptions = null
	): array|object {

		$urlBuilder = new ApiUrlBuilder(self::REWARDS_SITE_URI);
		$urlBuilder->withPathSite($siteUrl);
		$urlBuilder->withQueryParams($params);

		return $this->httpGet($urlBuilder, $requestOptions);
	}


	/**
	 * Get the list of achievements for the specified site
	 *
	 * @param string $siteUrl							The site for which to retrieve the list of achievements
	 * @param RewardGetAchievementListParams $params	The query parameters to be used with the request
	 * @param RequestOptions $requestOptions			The requestOptions to be used with the request
	 */
	public function getAchievements(
		string $siteUrl,
		RewardGetAchievementListParams $params = null,
		RequestOptions $requestOptions = null
	): array|object {

		$urlBuilder = new ApiUrlBuilder(self::ACHIEVEMENTS_SITE_URI);
		$urlBuilder->withPathSite($siteUrl);
		$urlBuilder->withQueryParams($params);

		return $this->httpGet($urlBuilder, $requestOptions);
	}


	/**
	 * Get the list of levels for the specified site
	 *
	 * @param string $siteUrl					The site for which to retrieve the list of levels
	 * @param RewardGetLevelListParams $params	The query parameters to be used with the request
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function getLevels(
		string $siteUrl,
		RewardGetLevelListParams $params = null,
		RequestOptions $requestOptions = null
	): array|object {

		$urlBuilder = new ApiUrlBuilder(self::LEVELS_SITE_URI);
		$urlBuilder->withPathSite($siteUrl);
		$urlBuilder->withQueryParams($params);

		return $this->httpGet($urlBuilder, $requestOptions);
	}


	/**
	 * Get the list of gifts for the specified site
	 *
	 * @param string $siteUrl					The site for which to retrieve the list of gifts
	 * @param RewardGetGiftListParams $params	The query parameters to be used with the request
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function getGifts(
		string $siteUrl,
		RewardGetGiftListParams $params = null,
		RequestOptions $requestOptions = null
	): array|object {

		$urlBuilder = new ApiUrlBuilder(self::GIFTS_SITE_URI);
		$urlBuilder->withPathSite($siteUrl);
		$urlBuilder->withQueryParams($params);

		return $this->httpGet($urlBuilder, $requestOptions);
	}


	/**
	 * Get the list of missions for the specified site
	 *
	 * @param string $siteUrl						The site for which to retrieve the list of missions
	 * @param RewardGetMissionListParams $params	The query parameters to be used with the request
	 * @param RequestOptions $requestOptions		The requestOptions to be used with the request
	 */
	public function getMissions(
		string $siteUrl,
		RewardGetMissionListParams $params = null,
		RequestOptions $requestOptions = null
	): array|object {

		$urlBuilder = new ApiUrlBuilder(self::MISSIONS_SITE_URI);
		$urlBuilder->withPathSite($siteUrl);
		$urlBuilder->withQueryParams($params);

		return $this->httpGet($urlBuilder, $requestOptions);
	}


	/**
	 * This method is used to add custom fields to an existing reward
	 *
	 * @param string $id						The ID of the reward to update
	 * @param CustomFieldValueRequestData $data	The list of custom fields to add
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function addCustomFields(string $id, CustomFieldValueRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		return $this->modCustomFields(HttpClient::POST, $id, $data, $requestOptions);
	}


	/**
	 * This method is used to update custom fields in an existing reward
	 *
	 * @param string $id						The ID of the reward to update
	 * @param CustomFieldValueRequestData $data	The list of custom fields to update
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function updateCustomFields(string $id, CustomFieldValueRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		return $this->modCustomFields(HttpClient::PUT, $id, $data, $requestOptions);
	}


	/**
	 * This method is used to delete custom fields in an existing reward
	 *
	 * @param string $id						The ID of the reward to update
	 * @param CustomFieldValueRequestData $data	The list of custom fields to update
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function deleteCustomFields(string $id, CustomFieldValueRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		return $this->modCustomFields(HttpClient::DELETE, $id, $data, $requestOptions);
	}


	/**
	 * Modifying custom fields
	 */
	private function modCustomFields(string $method, string $id, CustomFieldValueRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::REWARDS_CUSTOM_URI);
		$urlBuilder->withPathId($id);
		return $this->client->request($urlBuilder, $method, $data, $requestOptions);
	}


	/**
	 * This method is used to add a custom field with an image in an existing reward
	 *
	 * @param string $id						The ID of the reward to update
	 * @param CustomFieldValue $data	        The custom field to update
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function addCustomField(string $id, CustomFieldValue $data, string|false $image, RequestOptions $requestOptions = null): array|object
	{
		Preconditions::checkNotEmpty($image, "The image must not be empty");

		$urlBuilder = new ApiUrlBuilder(self::REWARDS_CUSTOM_IMAGE_URI);
		$urlBuilder->withPathId($id);

		return $this->httpUpload($urlBuilder, $image, $requestOptions, $data);
	}
}
