<?php
/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * The MamboPointsService class handles all Point related requests
 * to the Mambo API.
 */

declare(strict_types=1);

namespace Mambo\Point;

use Mambo\Point\Data\PointRequestData;
use Mambo\Point\Param\PointGetListParams;
use Mambo\Common\Service\AbstractService;
use Mambo\Common\Data\DeleteRequestData;
use Mambo\Common\Data\CustomFieldValueRequestData;
use Mambo\Common\Data\CustomFieldValue;
use Mambo\Http\ApiUrlBuilder;
use Mambo\Http\RequestOptions;
use Mambo\Http\HttpClient;
use Mambo\Util\Preconditions;

class PointsService extends AbstractService
{
	/**
	 * Point service end point URI
	 * @var string
	 */
	const POINTS_URI = "/v1/points";
	const POINTS_ID_URI = "/v1/points/{id}";
	const POINTS_IMAGE_URI = "/v1/points/{id}/image";
	const POINTS_CLONE_URI = "/v1/points/{id}/clone";
	const POINTS_CUSTOM_URI = "/v1/points/{id}/custom_fields";
	const POINTS_CUSTOM_IMAGE_URI = "/v1/points/{id}/custom_fields/image";
	const POINTS_TRANSACTIONS_BY_ACTIVITY_URI = "/v1/points/transactions/{id}";

	const POINTS_SITE_URI = "/v1/{site}/points";
	const POINTS_SITE_WALLETS_URI = "/v1/{site}/points/wallets";


	/**
	 * This method is used to create a new point.
	 *
	 * @param string $siteUrl					The site to which the point belongs to
	 * @param PointRequestData $data				The data sent to the API in order to create a point
	 * @param RequestOptions $requestOptions		The requestOptions to be used with the request
	 */
	public function create(string $siteUrl, PointRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::POINTS_SITE_URI);
		$urlBuilder->withPathSite($siteUrl);
		return $this->httpPost($urlBuilder, $requestOptions, $data);
	}


	/**
	 * Update an existing point by ID.
	 *
	 * @param string $id					The ID of the point to update
	 * @param PointRequestData $data	The data with which to update the specified point object
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function update(string $id, PointRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::POINTS_ID_URI);
		$urlBuilder->withPathId($id);
		return $this->httpPut($urlBuilder, $requestOptions, $data);
	}


	/**
	 * Upload an image for the point
	 *
	 * @param string $id		The point for which to upload the image
	 * @param string|false $image 	The image to upload for the point
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function uploadImage(string $id, string|false $image, RequestOptions $requestOptions = null): array|object
	{
		Preconditions::checkNotEmpty($image, "The image must not be empty");

		$urlBuilder = new ApiUrlBuilder(self::POINTS_IMAGE_URI);
		$urlBuilder->withPathId($id);

		return $this->httpUpload($urlBuilder, $image, $requestOptions);
	}


	/**
	 * Clone a point
	 *
	 * @param string $id		The ID of the point to clone
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function clonePoint(string $id, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::POINTS_CLONE_URI);
		$urlBuilder->withPathId($id);
		return $this->httpPost($urlBuilder, $requestOptions);
	}


	/**
	 * Delete a point by it's ID
	 *
	 * @param string $id				The ID of the point to delete
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function delete(string $id, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::POINTS_ID_URI);
		$urlBuilder->withPathId($id);
		return $this->httpDelete($urlBuilder, $requestOptions);
	}


	/**
	 * Delete a list of points by their ID
	 *
	 * @param DeleteRequestData $data		The list of IDs of the point to delete
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function deletePoints(DeleteRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::POINTS_URI);
		return $this->httpDelete($urlBuilder, $requestOptions, $data);
	}


	/**
	 * Get a point by it's ID
	 *
	 * @param string $id			The ID of the point to retrieve
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function get(string $id, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::POINTS_ID_URI);
		$urlBuilder->withPathId($id);
		return $this->httpGet($urlBuilder, $requestOptions);
	}


	/**
	 * Get the list of points for the specified site
	 *
	 * @param string $siteUrl					The site for which to retrieve the list of points
	 * @param PointGetListParams $params		The query parameters to be used with the request
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function getPoints(
		string $siteUrl,
		PointGetListParams $params = null,
		RequestOptions $requestOptions = null
	): array|object {

		$urlBuilder = new ApiUrlBuilder(self::POINTS_SITE_URI);
		$urlBuilder->withPathSite($siteUrl);
		$urlBuilder->withQueryParams($params);

		return $this->httpGet($urlBuilder, $requestOptions);
	}


	/**
	 * Get the list of point wallets for the specified site
	 *
	 * @param string $siteUrl					The site for which to retrieve the list of point wallets
	 * @param PointWalletsGetListParams $params	The query parameters to be used with the request
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function getPointWallets(
		string $siteUrl,
		PointWalletsGetListParams $params = null,
		RequestOptions $requestOptions = null
	): array|object {

		$urlBuilder = new ApiUrlBuilder(self::POINTS_SITE_WALLETS_URI);
		$urlBuilder->withPathSite($siteUrl);
		$urlBuilder->withQueryParams($params);

		return $this->httpGet($urlBuilder, $requestOptions);
	}


	/**
	 * Get transactions by Activity ID
	 *
	 * @param string $activityId				The ID of the activity for which to retrieve transactions
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function getTransactionsByActivityId(
		string $activityId,
		RequestOptions $requestOptions = null
	): array|object {
		$urlBuilder = new ApiUrlBuilder(self::POINTS_TRANSACTIONS_BY_ACTIVITY_URI);
		$urlBuilder->withPathId($activityId);
		return $this->httpGet($urlBuilder, $requestOptions);
	}


	/**
	 * This method is used to add custom fields to an existing point
	 *
	 * @param string $id						The ID of the point to update
	 * @param CustomFieldValueRequestData $data	The list of custom fields to add
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function addCustomFields(string $id, CustomFieldValueRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		return $this->modCustomFields(HttpClient::POST, $id, $data, $requestOptions);
	}


	/**
	 * This method is used to update custom fields in an existing point
	 *
	 * @param string $id						The ID of the point to update
	 * @param CustomFieldValueRequestData $data	The list of custom fields to update
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function updateCustomFields(string $id, CustomFieldValueRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		return $this->modCustomFields(HttpClient::PUT, $id, $data, $requestOptions);
	}


	/**
	 * This method is used to delete custom fields in an existing point
	 *
	 * @param string $id						The ID of the point to update
	 * @param CustomFieldValueRequestData $data	The list of custom fields to update
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function deleteCustomFields(string $id, CustomFieldValueRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		return $this->modCustomFields(HttpClient::DELETE, $id, $data, $requestOptions);
	}


	/**
	 * Modifying custom fields
	 */
	private function modCustomFields(string $method, string $id, CustomFieldValueRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::POINTS_CUSTOM_URI);
		$urlBuilder->withPathId($id);
		return $this->client->request($urlBuilder, $method, $data, $requestOptions);
	}


	/**
	 * This method is used to add a custom field with an image in an existing point
	 *
	 * @param string $id						The ID of the point to update
	 * @param CustomFieldValue $data         The custom field to update
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function addCustomField(string $id, CustomFieldValue $data, string|false $image, RequestOptions $requestOptions = null): array|object
	{
		Preconditions::checkNotEmpty($image, "The image must not be empty");

		$urlBuilder = new ApiUrlBuilder(self::POINTS_CUSTOM_IMAGE_URI);
		$urlBuilder->withPathId($id);

		return $this->httpUpload($urlBuilder, $image, $requestOptions, $data);
	}
}
