<?php
/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * This object captures the data required by the Point API in
 * order to create / update points.
 */

declare(strict_types=1);

namespace Mambo\Point\Data;

use Mambo\Common\Data\RequestData;
use Mambo\Common\Data\AbstractHasTagRequestData;
use Mambo\Common\Data\Recurrence\Recurrence;

class PointRequestData extends AbstractHasTagRequestData implements RequestData
{

	/**
	 * Whether the image associated to the Point should be removed
	 * @return boolean removeImage
	 */
	public function getRemoveImage()
	{
		return $this->data['removeImage'];
	}
	public function setRemoveImage($removeImage)
	{
		$this->data['removeImage'] = $removeImage;
	}

	/**
	 * The name of the point.
	 */
	public function getName()
	{
		return $this->data['name'];
	}
	public function setName($name)
	{
		$this->data['name'] = $name;
	}

	/**
	 * This flag indicates whether this point is active.
	 */
	public function getActive()
	{
		return $this->data['active'];
	}
	public function setActive($active)
	{
		$this->data['active'] = $active;
	}

	/**
	 * Indicates whether the point should be hidden in the summary
	 * of the Profile and Header widgets. This is usually associated
	 * to KPIs or points which are used only for tracking.
	 */
	public function getHideInSummary()
	{
		return $this->data['hideInSummary'];
	}
	public function setHideInSummary($hideInSummary)
	{
		$this->data['hideInSummary'] = $hideInSummary;
	}

	/**
	 * Indicates whether we should stop at zero when we are subtracting these
	 * points from a user. This will skip the user's balance validation as the
	 * points will never go below zero so the user is considered to always have
	 * sufficient balance.
	 * For example: the user has 5 points, we subtract 10 points, the user now
	 * has 0 points (instead of -5).
	 * See the points page in administration panel for more information.
	 */
	public function getStopAtZero()
	{
		return $this->data['stopAtZero'];
	}
	public function setStopAtZero($stopAtZero)
	{
		$this->data['stopAtZero'] = $stopAtZero;
	}

	/**
	 * Indicates whether the point can be set.
	 * See the points page in administration panel for more information.
	 */
	public function getSettable()
	{
		return $this->data['settable'];
	}
	public function setSettable($settable)
	{
		$this->data['settable'] = $settable;
	}


	/**
	 * Indicates whether the user can have negative points of this type.
	 * If the points cannot be negative then no user will ever have less
	 * than 0 points of this type.
	 * See the points page in administration panel for more information.
	 */
	public function getNegativeable()
	{
		return $this->data['negativeable'];
	}
	public function setNegativeable($negativeable)
	{
		$this->data['negativeable'] = $negativeable;
	}

	/**
	 * Indicates whether the points can be redeemed in exchange for items.
	 * See the points page in administration panel for more information.
	 */
	public function getRedeemable()
	{
		return $this->data['redeemable'];
	}
	public function setRedeemable($redeemable)
	{
		$this->data['redeemable'] = $redeemable;
	}

	/**
	 * Indicates whether the points can be gifted to other users.
	 * See the points page in administration panel for more information.
	 */
	public function getGiftable()
	{
		return $this->data['giftable'];
	}
	public function setGiftable($giftable)
	{
		$this->data['giftable'] = $giftable;
	}

	/**
	 * Indicates whether the points can be used as a bounty which will then
	 * be given to the user who meets the bounty's requirements.
	 * See the points page in administration panel for more information.
	 */
	public function getBountiable()
	{
		return $this->data['bountiable'];
	}
	public function setBountiable($bountiable)
	{
		$this->data['bountiable'] = $bountiable;
	}

	/**
	 * Indicates whether the points expire.
	 * See the points page in administration panel for more information.
	 */
	public function getExpirable()
	{
		return $this->data['expirable'];
	}
	public function setExpirable($expirable)
	{
		$this->data['expirable'] = $expirable;
	}

	/**
	 * If the points are expiration then this field should contain the general
	 * expiration information.
	 * See the {@link NeverExpiration}, {@link FixedPeriodExpiration} or
	 * {@link VariablePeriodExpiration} objects for more information.
	 */
	public function getExpiration()
	{
		return $this->data['expiration'];
	}
	public function setExpiration(Recurrence $expiration)
	{
		$this->data['expiration'] = $expiration;
	}

	/**
	 * The description associated with the point.
	 * See the points page in administration panel for more information.
	 */
	public function getDescription()
	{
		return $this->data['description'];
	}
	public function setDescription($description)
	{
		$this->data['description'] = $description;
	}

	/**
	 * This contains the point's decimal places information.
	 * Note that this value can only be set when the point is being
	 * created so choose wisely.
	 * See the {@link Decimals} object for more information.
	 */
	public function getDecimals()
	{
		return $this->data['decimals'];
	}
	public function setDecimals(Decimals $decimals)
	{
		$this->data['decimals'] = $decimals;
	}

	/**
	 * This contains the point's units of measure.
	 * See the {@link Units} object for more information.
	 */
	public function getUnits()
	{
		return $this->data['units'];
	}
	public function setUnits(Units $units)
	{
		$this->data['units'] = $units;
	}

	/**
	 * Custom fields defined for the point. These can contain additional
	 * data or any kind of information you would like to store which isn't a
	 * standard field of the point.
	 */
	public function getCustomFields()
	{
		return $this->data['customFields'];
	}
	public function setCustomFields(array $customFields)
	{
		$this->data['customFields'] = $customFields;
	}

	/**
	 * This contains the list of the translations which must be used with the object.
	 * Setting a property directly on the object (such as using setName()) will add
	 * the property to the list of translations using the default language of the server.
	 * If you set a property on the object directly (such as setName()) and then set
	 * the same property using setTranslations(), then the setTranslations() property
	 * will take priority.
	 */
	public function getTranslations()
	{
		return $this->data['translations'];
	}
	public function setTranslations(array $translations)
	{
		$this->data['translations'] = $translations;
	}
	public function addTranslation(PointTranslation $translation)
	{
		if (!isset($this->data['translations'])) {
			$this->data['translations'] = array();
		}
		array_push($this->data['translations'], $translation);
	}


	/**
	 * Return the JSON string equivalent of this object
	 */
	public function getJsonString(): string
	{
		$json = $this->data;

		if (isset($json['expiration'])) {
			$json['expiration'] = $json['expiration']->getJsonArray();
		}

		if (isset($json['units'])) {
			$json['units'] = $json['units']->getJsonArray();
		}

		if (isset($json['decimals'])) {
			$json['decimals'] = $json['decimals']->getJsonArray();
		}

		if (isset($json['customFields'])) {
			$customFieldsArr = array();
			foreach ($json['customFields'] as $customField) {
				array_push($customFieldsArr, $customField->getJsonArray());
			}
			$json['customFields'] = $customFieldsArr;
		}

		if (isset($json['translations'])) {
			$translationsArr = array();
			foreach ($json['translations'] as $translation) {
				array_push($translationsArr, $translation->getJsonArray());
			}
			$json['translations'] = $translationsArr;
		}

		return json_encode($json);
	}
}
