<?php
/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * The MamboNotificationsService class handles all Notification related requests
 * to the Mambo API.
 */

declare(strict_types=1);

namespace Mambo\Notification;

use Mambo\Notification\Data\ClearNotificationsRequestData;
use Mambo\Notification\Param\NotificationGetByActivityIdParams;
use Mambo\Notification\Param\NotificationGetParams;
use Mambo\Notification\Param\NotificationGetListParams;
use Mambo\Common\Service\AbstractService;
use Mambo\Http\ApiUrlBuilder;
use Mambo\Http\RequestOptions;

class NotificationsService extends AbstractService
{
	/**
	 * Notification service end point URI
	 * @var string
	 */
	const NOTIFICATIONS_ACTIVITY_ID_URI = "/v1/notifications/activity/{id}";
	const NOTIFICATIONS_ID_URI = "/v1/notifications/{id}";
	const CLEAR_NOTIFICATIONS_ID_URI = "/v1/notifications/{id}/clear";
	const CLEAR_NOTIFICATIONS_URI = "/v1/notifications/clear";

	const NOTIFICATIONS_SITE_URI = "/v1/{site}/points";


	/**
	 * Clear a notification by it's ID
	 *
	 * @param string $id		The ID of the notification to clear
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function clearNotification(string $id, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::CLEAR_NOTIFICATIONS_ID_URI);
		$urlBuilder->withPathId($id);
		return $this->httpPost($urlBuilder, $requestOptions);
	}


	/**
	 * Clear a list of notifications by their ID
	 *
	 * @param ClearNotificationsRequestData $data		The list of IDs of the notification to clear
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function clearNotifications(ClearNotificationsRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::CLEAR_NOTIFICATIONS_URI);
		return $this->httpPost($urlBuilder, $requestOptions, $data);
	}


	/**
	 * Get a notification by it's ID
	 *
	 * @param string $id						The ID of the notification to retrieve
	 * @param NotificationGetParams $params		The query parameters to be used with the request
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function get(string $id, NotificationGetParams $params = null, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::NOTIFICATIONS_ID_URI);
		$urlBuilder->withPathId($id);
		$urlBuilder->withQueryParams($params);

		return $this->httpGet($urlBuilder, $requestOptions);
	}


	/**
	 * Get notifications by Activity ID
	 *
	 * @param string $activityId						The ID of the activity for which to retrieve notifications
	 * @param NotificationGetByActivityIdParams $params	The query parameters to be used with the request
	 * @param RequestOptions $requestOptions			The requestOptions to be used with the request
	 */
	public function getByActivityId(
		string $activityId,
		NotificationGetByActivityIdParams $params = null,
		RequestOptions $requestOptions = null
	): array|object {

		$urlBuilder = new ApiUrlBuilder(self::NOTIFICATIONS_ACTIVITY_ID_URI);
		$urlBuilder->withPathId($activityId);
		$urlBuilder->withQueryParams($params);

		return $this->httpGet($urlBuilder, $requestOptions);
	}


	/**
	 * Get the list of notifications for the specified site
	 *
	 * @param string $siteUrl					The site for which to retrieve the list of notifications
	 * @param NotificationGetListParams $params	The query parameters to be used with the request
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function getNotifications(
		string $siteUrl,
		NotificationGetListParams $params = null,
		RequestOptions $requestOptions = null
	): array|object {

		$urlBuilder = new ApiUrlBuilder(self::NOTIFICATIONS_SITE_URI);
		$urlBuilder->withPathSite($siteUrl);
		$urlBuilder->withQueryParams($params);

		return $this->httpGet($urlBuilder, $requestOptions);
	}
}
