<?php
/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * Properties used to configuration the HttpClient
 */

declare(strict_types=1);

namespace Mambo;

use Mambo\ClientConfiguration;
use Mambo\Activity\ActivitiesService;
use Mambo\Analytics\AnalyticsService;
use Mambo\Behaviour\BehavioursService;
use Mambo\Coupon\CouponsService;
use Mambo\CustomField\CustomFieldsService;
use Mambo\DataStore\DataStoresService;
use Mambo\Http\MamboCredentials;
use Mambo\Http\HttpClient;
use Mambo\Kpi\KpisService;
use Mambo\Language\LanguagesService;
use Mambo\Leaderboard\LeaderboardsService;
use Mambo\Notification\NotificationsService;
use Mambo\Point\PointsService;
use Mambo\Reward\RewardsService;
use Mambo\Site\SitesService;
use Mambo\Tag\TagsService;
use Mambo\User\UsersService;
use Mambo\Webhook\WebhooksService;
use Mambo\Util\Preconditions;

class MamboClient
{
	/** @var HttpClient */
	private $httpClient;

	/**
	 * @param string $publicKey
	 * @param string $privateKey
	 * @throws \InvalidArgumentException
	 */
	public function __construct(string $publicKey, string $privateKey)
	{
		Preconditions::checkNotEmpty($publicKey, 'publicKey must not be empty');
		Preconditions::checkNotEmpty($privateKey, 'privateKey must not be empty');

		$this->httpClient = new HttpClient(
			new MamboCredentials($publicKey, $privateKey),
			new ClientConfiguration()
		);
	}

	/**
	 * @param ClientConfiguration $config
	 * @throws \InvalidArgumentException
	 */
	public static function withConfiguration(ClientConfiguration $config): self
	{
		Preconditions::checkNotNull($config->getCredentials(), 'credentials must not be null');

		$instance = new self($config->getCredentials()->getPublicKey(), $config->getCredentials()->getPrivateKey());
		$instance->httpClient = new HttpClient($config->getCredentials(), $config);
		return $instance;
	}

	public function activities(): ActivitiesService
	{
		return new ActivitiesService($this->httpClient);
	}

	public function analytics(): AnalyticsService
	{
		return new AnalyticsService($this->httpClient);
	}

	public function behaviours(): BehavioursService
	{
		return new BehavioursService($this->httpClient);
	}

	public function coupons(): CouponsService
	{
		return new CouponsService($this->httpClient);
	}

	public function customFields(): CustomFieldsService
	{
		return new CustomFieldsService($this->httpClient);
	}

	public function dataStores(): DataStoresService
	{
		return new DataStoresService($this->httpClient);
	}

	public function kpis(): KpisService
	{
		return new KpisService($this->httpClient);
	}

	public function languages(): LanguagesService
	{
		return new LanguagesService($this->httpClient);
	}

	public function leaderboards(): LeaderboardsService
	{
		return new LeaderboardsService($this->httpClient);
	}

	public function notifications(): NotificationsService
	{
		return new NotificationsService($this->httpClient);
	}

	public function points(): PointsService
	{
		return new PointsService($this->httpClient);
	}

	public function rewards(): RewardsService
	{
		return new RewardsService($this->httpClient);
	}

	public function sites(): SitesService
	{
		return new SitesService($this->httpClient);
	}

	public function tags(): TagsService
	{
		return new TagsService($this->httpClient);
	}

	public function users(): UsersService
	{
		return new UsersService($this->httpClient);
	}

	public function webhooks(): WebhooksService
	{
		return new WebhooksService($this->httpClient);
	}

	public function getVersion(): string
	{
		return $this->httpClient->getVersion();
	}
}
