<?php
/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * The MamboLeaderboardsService class handles all Leaderboard related requests
 * to the Mambo API.
 */

declare(strict_types=1);

namespace Mambo\Leaderboard;

use Mambo\Leaderboard\Data\LeaderboardRequestData;
use Mambo\Leaderboard\Param\LeaderboardGetListParams;
use Mambo\Leaderboard\Param\LeaderboardGetParams;
use Mambo\Common\Service\AbstractService;
use Mambo\Common\Data\DeleteRequestData;
use Mambo\Http\ApiUrlBuilder;
use Mambo\Http\RequestOptions;

class LeaderboardsService extends AbstractService
{
	/**
	 * Leaderboard service end point URI
	 * @var string
	 */
	const LEADERBOARDS_SITE_URI = "/v1/{site}/leaderboards";

	const LEADERBOARDS_URI = "/v1/leaderboards";
	const LEADERBOARDS_ID_URI = "/v1/leaderboards/{id}";
	const LEADERBOARDS_CLONE_URI = "/v1/leaderboards/{id}/clone";
	const LEADERBOARDS_REGENERATE_URI = "/v1/leaderboards/{id}/regenerate";
	const LEADERBOARDS_RANK_URI = "/v1/leaderboards/{id}/rank";

	/**
	 * This method is used to create a new leaderboard.
	 *
	 * @param string $siteUrl					The site to which the leaderboard belongs to
	 * @param LeaderboardRequestData $data		The data sent to the API in order to create a leaderboard
	 * @param RequestOptions $requestOptions		The requestOptions to be used with the request
	 */
	public function create(string $siteUrl, LeaderboardRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::LEADERBOARDS_SITE_URI);
		$urlBuilder->withPathSite($siteUrl);
		return $this->httpPost($urlBuilder, $requestOptions, $data);
	}


	/**
	 * Update an existing leaderboard by ID.
	 *
	 * @param string $id						The ID of the leaderboard to update
	 * @param LeaderboardRequestData $data	The data with which to update the specified leaderboard object
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function update(string $id, LeaderboardRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::LEADERBOARDS_ID_URI);
		$urlBuilder->withPathId($id);
		return $this->httpPut($urlBuilder, $requestOptions, $data);
	}


	/**
	 * Clone a leaderboard
	 *
	 * @param string $id		The ID of the leaderboard to clone
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function cloneLeaderboard(string $id, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::LEADERBOARDS_CLONE_URI);
		$urlBuilder->withPathId($id);
		return $this->httpPost($urlBuilder, $requestOptions);
	}


	/**
	 * Regenerate a leaderboard's score board from the ground up.
	 *
	 * @param string $id		The ID of the leaderboard to regenerate
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function regenerate(string $id, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::LEADERBOARDS_REGENERATE_URI);
		$urlBuilder->withPathId($id);
		return $this->httpPost($urlBuilder, $requestOptions);
	}


	/**
	 * Rank a leaderboard's users.
	 *
	 * @param string $id		The ID of the leaderboard to rank
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function rank(string $id, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::LEADERBOARDS_RANK_URI);
		$urlBuilder->withPathId($id);
		return $this->httpPost($urlBuilder, $requestOptions);
	}


	/**
	 * Delete a leaderboard by it's ID
	 *
	 * @param string $id				The ID of the leaderboard to delete
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function delete(string $id, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::LEADERBOARDS_ID_URI);
		$urlBuilder->withPathId($id);
		return $this->httpDelete($urlBuilder, $requestOptions);
	}


	/**
	 * Delete a list of leaderboards by their ID
	 *
	 * @param DeleteRequestData $data		The list of IDs of the leaderboard to delete
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function deleteLeaderboards(DeleteRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::LEADERBOARDS_URI);
		return $this->httpDelete($urlBuilder, $requestOptions, $data);
	}


	/**
	 * Get the leaderboard specified by site and it's ID. To see a list of the leaderboard
	 * IDs, use the getLeaderboards() method.
	 *
	 * @param string $id						The ID of the leaderboard to retrieve
	 * @param LeaderboardGetParams $params		The query parameters to be used with the request
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function getLeaderboard(
		string $id,
		LeaderboardGetParams $params = null,
		RequestOptions $requestOptions = null
	): array|object {

		$urlBuilder = new ApiUrlBuilder(self::LEADERBOARDS_ID_URI);
		$urlBuilder->withPathId($id);
		$urlBuilder->withQueryParams($params);

		return $this->httpGet($urlBuilder, $requestOptions);
	}


	/**
	 * Get the list of leaderboards for the specified site.
	 * Note: the leaderboards returned using this method are not populated
	 * with the users. In order to retrieve a leaderboard with it's users
	 * you should call one of the other methods.
	 *
	 * @param string $siteUrl					The site for which to retrieve the list of leaderboards
	 * @param LeaderboardGetListParams $params	The query parameters to be used with the request
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function getLeaderboards(
		string $siteUrl,
		LeaderboardGetListParams $params = null,
		RequestOptions $requestOptions = null
	): array|object {

		$urlBuilder = new ApiUrlBuilder(self::LEADERBOARDS_SITE_URI);
		$urlBuilder->withPathSite($siteUrl);
		$urlBuilder->withQueryParams($params);

		return $this->httpGet($urlBuilder, $requestOptions);
	}
}
