<?php
/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * This object captures the data required by the Event API in
 * order to track a behaviour performed by a specific user.
 */

declare(strict_types=1);

namespace Mambo\Leaderboard\Data;

use Mambo\Common\Data\RequestData;
use Mambo\Common\Data\AbstractHasTagRequestData;

class LeaderboardRequestData extends AbstractHasTagRequestData implements RequestData
{

	/**
	 * The name of the leaderboard. See the leaderboard page in
	 * administration panel for more information.
	 */
	public function getName()
	{
		return $this->data['name'];
	}
	public function setName($name)
	{
		$this->data['name'] = $name;
	}

	/**
	 * Indicates whether the leaderboard is active or not.
	 */
	public function getActive()
	{
		return $this->data['active'];
	}
	public function setActive($active)
	{
		$this->data['active'] = $active;
	}

	/**
	 * This represents the date from which this leaderboard is active.
	 * If no date is specified, the leaderboard is always active.
	 * This must be a timestamp in ISO 8601 format with
	 * millisecond precision: YYYY-MM-DDTHH:MM:SS.MMMZ.
	 */
	public function getStartDate()
	{
		return $this->data['startDate'];
	}
	public function setStartDate($startDate)
	{
		$this->data['startDate'] = $startDate;
	}

	/**
	 * This represents the date from which this leaderboard is no longer active.
	 * If no date is specified, the leaderboard is always active.
	 * This must be a timestamp in ISO 8601 format with
	 * millisecond precision: YYYY-MM-DDTHH:MM:SS.MMMZ.
	 */
	public function getEndDate()
	{
		return $this->data['endDate'];
	}
	public function setEndDate($endDate)
	{
		$this->data['endDate'] = $endDate;
	}

	/**
	 * Determines which time periods the ranking will take place for.
	 * Only specified periods will be ranked and retrievable via the APIs.
	 * Note that if this property is empty, the leaderboard will be unranked.
	 * Valid values include: day, week, month, all
	 */
	public function getRankBy()
	{
		return $this->data['rankBy'];
	}
	public function setRankBy(array $rankBy)
	{
		$this->data['rankBy'] = $rankBy;
	}

	/**
	 * Indicates whether the leaderboard should be in the reverse order.
	 * This will place the smallest values at the top of the leaderboard.
	 */
	public function getIsReverse()
	{
		return $this->data['isReverse'];
	}
	public function setIsReverse($isReverse)
	{
		$this->data['isReverse'] = $isReverse;
	}

	/**
	 * Indicates whether the lowest leaderboard score should be zero or
	 * whether the leaderboard can have a negative score.
	 */
	public function getStopAtZero()
	{
		return $this->data['stopAtZero'];
	}
	public function setStopAtZero($stopAtZero)
	{
		$this->data['stopAtZero'] = $stopAtZero;
	}


	/**
	 * Indicates whether users in the leaderboard should remain in the leaderboard
	 * when their last activity becomes older than the period being tracked.
	 */
	public function getIncludeInactiveUsers()
	{
		return $this->data['includeInactiveUsers'];
	}
	public function setIncludeInactiveUsers($includeInactiveUsers)
	{
		$this->data['includeInactiveUsers'] = $includeInactiveUsers;
	}


	/**
	 * Indicates whether the leaderboard should be automatically filtered
	 * by the tag even if the tag is not explicitly defined in the API call.
	 * In order for auto filtering to work, there must only be one tag defined
	 * in the filterableByTagIds list. This tag will be used as the one to
	 * auto filter the leaderboard with.
	 */
	public function getAutoFilterByTag()
	{
		return $this->data['autoFilterByTag'];
	}
	public function setAutoFilterByTag($autoFilterByTag)
	{
		$this->data['autoFilterByTag'] = $autoFilterByTag;
	}

	/**
	 * This contains the list of the tag IDs by which this
	 * leaderboard can be filtered.
	 */
	public function getFilterableByTagIds()
	{
		return $this->data['filterableByTagIds'];
	}
	public function setFilterableByTagIds(array $filterableByTagIds)
	{
		$this->data['filterableByTagIds'] = $filterableByTagIds;
	}

	/**
	 * This contains the list of the tag IDs such that any user with at least
	 * one of those tags is excluded from the leaderboard.
	 */
	public function getExcludedUsersTagIds()
	{
		return $this->data['excludedUsersTagIds'];
	}
	public function setExcludedUsersTagIds(array $excludedUsersTagIds)
	{
		$this->data['excludedUsersTagIds'] = $excludedUsersTagIds;
	}

	/**
	 * This must contain the list of the IDs of the points which must
	 * be added together for this leaderboard score.
	 */
	public function getPointIds()
	{
		return $this->data['pointIds'];
	}
	public function setPointIds(array $pointIds)
	{
		$this->data['pointIds'] = $pointIds;
	}
	public function addPointId($pointId)
	{
		if (!isset($this->data['pointIds'])) {
			$this->data['pointIds'] = array();
		}
		array_push($this->data['pointIds'], (string) $pointId);
	}

	/**
	 * The attributes of the leaderboard. There are currently two types of
	 * attributes: LeaderboardBehaviourAttrs and LeaderboardPointAttrs.
	 */
	public function getAttrs()
	{
		return $this->data['attrs'];
	}
	public function setAttrs($attrs)
	{
		$this->data['attrs'] = $attrs;
	}


	/**
	 * This contains the list of the translations which must be used with the object.
	 * Setting a property directly on the object (such as using setName()) will add
	 * the property to the list of translations using the default language of the server.
	 * If you set a property on the object directly (such as setName()) and then set
	 * the same property using setTranslations(), then the setTranslations() property
	 * will take priority.
	 */
	public function getTranslations()
	{
		return $this->data['translations'];
	}
	public function setTranslations(array $translations)
	{
		$this->data['translations'] = $translations;
	}
	public function addTranslation(LeaderboardTranslation $translation)
	{
		if (!isset($this->data['translations'])) {
			$this->data['translations'] = array();
		}
		array_push($this->data['translations'], $translation);
	}


	/**
	 * Return the JSON string equivalent of this object
	 */
	public function getJsonString(): string
	{
		$json = $this->data;

		if (isset($json['attrs'])) {
			$json['attrs'] = $json['attrs']->getJsonArray();
		}

		if (isset($json['translations'])) {
			$translationsArr = array();
			foreach ($json['translations'] as $translation) {
				array_push($translationsArr, $translation->getJsonArray());
			}
			$json['translations'] = $translationsArr;
		}

		return json_encode($json);
	}
}
