<?php
/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * The MamboLanguagesService class handles all Language related requests
 * to the Mambo API.
 */

declare(strict_types=1);

namespace Mambo\Language;

use Mambo\Language\Data\LanguageRequestData;
use Mambo\Common\Service\AbstractService;
use Mambo\Common\Data\DeleteRequestData;
use Mambo\Http\ApiUrlBuilder;
use Mambo\Http\RequestOptions;

class LanguagesService extends AbstractService
{
	/**
	 * Language service end language URI
	 * @var string
	 */
	const LANGUAGES_URI = "/v1/languages";
	const LANGUAGES_ID_URI = "/v1/languages/{id}";
	const LANGUAGES_CLONE_URI = "/v1/languages/{id}/clone";

	const LANGUAGES_SITE_URI = "/v1/{site}/languages";


	/**
	 * This method is used to create a new language.
	 *
	 * @param string $siteUrl					The site to which the language belongs to
	 * @param LanguageRequestData $data			The data sent to the API in order to create a language
	 * @param RequestOptions $requestOptions		The requestOptions to be used with the request
	 */
	public function create(string $siteUrl, LanguageRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::LANGUAGES_SITE_URI);
		$urlBuilder->withPathSite($siteUrl);
		return $this->httpPost($urlBuilder, $requestOptions, $data);
	}


	/**
	 * Update an existing language by ID.
	 *
	 * @param string $id					The ID of the language to update
	 * @param LanguageRequestData $data	The data with which to update the specified language object
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function update(string $id, LanguageRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::LANGUAGES_ID_URI);
		$urlBuilder->withPathId($id);
		return $this->httpPut($urlBuilder, $requestOptions, $data);
	}


	/**
	 * Clone a language
	 *
	 * @param string $id		The ID of the language to clone
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function cloneLanguage(string $id, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::LANGUAGES_CLONE_URI);
		$urlBuilder->withPathId($id);
		return $this->httpPost($urlBuilder, $requestOptions);
	}


	/**
	 * Delete a language by it's ID
	 *
	 * @param string $id				The ID of the language to delete
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function delete(string $id, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::LANGUAGES_ID_URI);
		$urlBuilder->withPathId($id);
		return $this->httpDelete($urlBuilder, $requestOptions);
	}


	/**
	 * Delete a list of languages by their ID
	 *
	 * @param DeleteRequestData $data		The list of IDs of the language to delete
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function deleteLanguages(DeleteRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::LANGUAGES_URI);
		return $this->httpDelete($urlBuilder, $requestOptions, $data);
	}


	/**
	 * Get a language by it's ID
	 *
	 * @param string $id			The ID of the language to retrieve
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function get(string $id, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::LANGUAGES_ID_URI);
		$urlBuilder->withPathId($id);
		return $this->httpGet($urlBuilder, $requestOptions);
	}


	/**
	 * Get the list of languages for the specified site
	 *
	 * @param string $siteUrl	The site for which to retrieve the list of languages
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function getLanguages(string $siteUrl, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::LANGUAGES_SITE_URI);
		$urlBuilder->withPathSite($siteUrl);
		return $this->httpGet($urlBuilder, $requestOptions);
	}
}
