<?php
/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * The MamboKPIsService class handles all KPI related requests
 * to the Mambo API.
 */

declare(strict_types=1);

namespace Mambo\Kpi;

use Mambo\Kpi\Data\KpiRequestData;
use Mambo\Kpi\Data\KpiTargetRequestData;
use Mambo\Kpi\Param\KpiGetListParams;
use Mambo\Kpi\Param\KpiGetStatesParams;
use Mambo\Common\Service\AbstractService;
use Mambo\Common\Data\DeleteRequestData;
use Mambo\Common\Data\CustomFieldValueRequestData;
use Mambo\Common\Data\CustomFieldValue;
use Mambo\Http\ApiUrlBuilder;
use Mambo\Http\RequestOptions;
use Mambo\Http\HttpClient;
use Mambo\Util\Preconditions;

class KpisService extends AbstractService
{
	/**
	 * KPI service end point URIs
	 * @var string
	 */
	const KPIS_URI = "/v1/kpis";
	const KPIS_ID_URI = "/v1/kpis/{id}";
	const KPIS_IMAGE_URI = "/v1/kpis/{id}/image";
	const KPIS_CLONE_URI = "/v1/kpis/{id}/clone";
	const KPIS_CUSTOM_URI = "/v1/kpis/{id}/custom_fields";
	const KPIS_CUSTOM_IMAGE_URI = "/v1/kpis/{id}/custom_fields/image";

	const KPIS_STATES_URI = "/v1/kpis/{id}/states";
	const KPIS_TARGETS_URI = "/v1/kpis/{id}/targets";
	const KPIS_TARGETS_ID_URI = "/v1/kpis/targets/{id}";
	const KPIS_TARGETS_NO_ID_URI = "/v1/kpis/targets";
	const KPIS_SITE_URI = "/v1/{site}/kpis";
	const KPIS_UUID_URI = "/v1/{site}/kpis/{uuid}";


	/**
	 * This method is used to create a new KPI.
	 *
	 * @param string $siteUrl				The site to which the KPI belongs to
	 * @param KpiRequestData $data			The data sent to the API in order to create a KPI
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function create(string $siteUrl, KpiRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::KPIS_SITE_URI);
		$urlBuilder->withPathSite($siteUrl);
		return $this->httpPost($urlBuilder, $requestOptions, $data);
	}


	/**
	 * Update an existing KPI by ID.
	 *
	 * @param string $id					The ID of the KPI to update
	 * @param KpiRequestData $data	The data with which to update the specified KPI object
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function update(string $id, KpiRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::KPIS_ID_URI);
		$urlBuilder->withPathId($id);
		return $this->httpPut($urlBuilder, $requestOptions, $data);
	}


	/**
	 * Upload an image for the KPI
	 *
	 * @param string $id		The KPI for which to upload the image
	 * @param string|false $image 	The image to upload for the KPI
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function uploadImage(string $id, string|false $image, RequestOptions $requestOptions = null): array|object
	{
		Preconditions::checkNotEmpty($image, "The image must not be empty");

		$urlBuilder = new ApiUrlBuilder(self::KPIS_IMAGE_URI);
		$urlBuilder->withPathId($id);

		return $this->httpUpload($urlBuilder, $image, $requestOptions);
	}


	/**
	 * Clone a KPI
	 *
	 * @param string $id		The ID of the KPI to clone
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function cloneKpi(string $id, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::KPIS_CLONE_URI);
		$urlBuilder->withPathId($id);
		return $this->httpPost($urlBuilder, $requestOptions);
	}


	/**
	 * Delete a KPI by it's ID
	 *
	 * @param string $id				The ID of the KPI to delete
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function delete(string $id, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::KPIS_ID_URI);
		$urlBuilder->withPathId($id);
		return $this->httpDelete($urlBuilder, $requestOptions);
	}


	/**
	 * Delete a list of KPIs by their ID
	 *
	 * @param DeleteRequestData $data		The list of IDs of the KPI to delete
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function deleteKpis(DeleteRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::KPIS_URI);
		return $this->httpDelete($urlBuilder, $requestOptions, $data);
	}


	/**
	 * Get a KPI by it's ID
	 *
	 * @param string $id			The ID of the KPI to retrieve
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function get(string $id, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::KPIS_ID_URI);
		$urlBuilder->withPathId($id);
		return $this->httpGet($urlBuilder, $requestOptions);
	}


	/**
	 * Get the list of KPIs for the specified site
	 *
	 * @param string $siteUrl					The site for which to retrieve the list of KPIs
	 * @param KpiGetListParams $params			The query parameters to be used with the request
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function getKpis(
		string $siteUrl,
		KpiGetListParams $params = null,
		RequestOptions $requestOptions = null
	): array|object {

		$urlBuilder = new ApiUrlBuilder(self::KPIS_SITE_URI);
		$urlBuilder->withPathSite($siteUrl);
		$urlBuilder->withQueryParams($params);

		return $this->httpGet($urlBuilder, $requestOptions);
	}


	/**
	 * Get the list of KPIs associated to the user specified.
	 *
	 * @param string $siteUrl					The site for which to retrieve the list of KPIs
	 * @param string $uuid	    				The UUID of the user for whom we want to fetch the KPIs
	 * @param KpiGetStatesParams $params		The query parameters to be used with the request
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function getKpiStatesByUser(
		string $siteUrl,
		string $uuid,
		KpiGetStatesParams $params = null,
		RequestOptions $requestOptions = null
	): array|object {

		$urlBuilder = new ApiUrlBuilder(self::KPIS_UUID_URI);
		$urlBuilder->withPathSite($siteUrl);
		$urlBuilder->withPathUuid($uuid);
		$urlBuilder->withQueryParams($params);

		return $this->httpGet($urlBuilder, $requestOptions);
	}


	/**
	 * Get the list of user states assigned to this KPI
	 *
	 * @param string $id						The KPI for which to retrieve the list of states
	 * @param KpiGetStatesParams $params		The query parameters to be used with the request
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function getKpiStates(
		string $id,
		KpiGetStatesParams $params = null,
		RequestOptions $requestOptions = null
	): array|object {

		$urlBuilder = new ApiUrlBuilder(self::KPIS_STATES_URI);
		$urlBuilder->withPathId($id);
		$urlBuilder->withQueryParams($params);

		return $this->httpGet($urlBuilder, $requestOptions);
	}


	/**
	 * This method is used to create a new KPI target.
	 *
	 * @param string $kpiId         				The KPI to which the target belongs to
	 * @param KpiTargetRequestData $data			The data sent to the API in order to create a KPI target
	 * @param RequestOptions $requestOptions		The requestOptions to be used with the request
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function createKpiTarget(string $kpiId, KpiTargetRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::KPIS_TARGETS_URI);
		$urlBuilder->withPathId($kpiId);
		return $this->httpPost($urlBuilder, $requestOptions, $data);
	}


	/**
	 * Update an existing KPI target by ID.
	 *
	 * @param string $id					The ID of the KPI target to update
	 * @param KpiTargetRequestData $data	The data with which to update the specified KPI object
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function updateKpiTarget(string $id, KpiTargetRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::KPIS_TARGETS_ID_URI);
		$urlBuilder->withPathId($id);
		return $this->httpPut($urlBuilder, $requestOptions, $data);
	}


	/**
	 * Delete a list of KPI targets by their ID
	 *
	 * @param DeleteRequestData $data	The list of IDs of the KPI targets to delete
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function deleteKpiTargets(DeleteRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::KPIS_TARGETS_NO_ID_URI);
		return $this->httpDelete($urlBuilder, $requestOptions, $data);
	}


	/**
	 * Get the list of KPI targets for the specified KPI
	 *
	 * @param string $kpiId	   The KPI for which to retrieve the list of targets
	 * @param integer $page		Specifies the page of results to retrieve
	 * @param integer $count		Specifies the number of results to retrieve, up to a maximum of 100
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function getKpiTargets(string $kpiId, $page = null, $count = null, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::KPIS_TARGETS_URI);
		$urlBuilder->withPathId($kpiId);

		return $this->httpGet($urlBuilder, $requestOptions);
	}


	/**
	 * This method is used to add custom fields to an existing KPI
	 *
	 * @param string $id						The ID of the KPI to update
	 * @param CustomFieldValueRequestData $data	The list of custom fields to add
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function addCustomFields(string $id, CustomFieldValueRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		return $this->modCustomFields(HttpClient::POST, $id, $data, $requestOptions);
	}


	/**
	 * This method is used to update custom fields in an existing KPI
	 *
	 * @param string $id						The ID of the KPI to update
	 * @param CustomFieldValueRequestData $data	The list of custom fields to update
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function updateCustomFields(string $id, CustomFieldValueRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		return $this->modCustomFields(HttpClient::PUT, $id, $data, $requestOptions);
	}


	/**
	 * This method is used to delete custom fields in an existing KPI
	 *
	 * @param string $id						The ID of the KPI to update
	 * @param CustomFieldValueRequestData $data	The list of custom fields to update
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function deleteCustomFields(string $id, CustomFieldValueRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		return $this->modCustomFields(HttpClient::DELETE, $id, $data, $requestOptions);
	}


	/**
	 * Modifying custom fields
	 */
	private function modCustomFields(string $method, string $id, CustomFieldValueRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::KPIS_CUSTOM_URI);
		$urlBuilder->withPathId($id);
		return $this->client->request($urlBuilder, $method, $data, $requestOptions);
	}


	/**
	 * This method is used to add a custom field with an image in an existing KPI
	 *
	 * @param string $id						The ID of the KPI to update
	 * @param CustomFieldValue $data	        The custom field to update
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function addCustomField(string $id, CustomFieldValue $data, string|false $image, RequestOptions $requestOptions = null): array|object
	{
		Preconditions::checkNotEmpty($image, "The image must not be empty");

		$urlBuilder = new ApiUrlBuilder(self::KPIS_CUSTOM_IMAGE_URI);
		$urlBuilder->withPathId($id);

		return $this->httpUpload($urlBuilder, $image, $requestOptions, $data);
	}
}
