<?php
/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * This class encapsulates data related to a KPIs target
 */

declare(strict_types=1);

namespace Mambo\Kpi\Data;

use Mambo\Common\Data\RequestData;

class KpiTargetRequestData implements RequestData
{
	private $data = array();


	/**
	 * Defines the KPIs daily target
	 */
	public function getDaily()
	{
		return $this->data['daily'];
	}
	public function setDaily($daily)
	{
		$this->data['daily'] = $daily;
	}


	/**
	 * Defines the KPIs weekly target
	 */
	public function getWeekly()
	{
		return $this->data['weekly'];
	}
	public function setWeekly($weekly)
	{
		$this->data['weekly'] = $weekly;
	}


	/**
	 * Defines the KPIs monthly target
	 */
	public function getMonthly()
	{
		return $this->data['monthly'];
	}
	public function setMonthly($monthly)
	{
		$this->data['monthly'] = $monthly;
	}


	/**
	 * Defines the KPIs quarterly target
	 */
	public function getQuarterly()
	{
		return $this->data['quarterly'];
	}
	public function setQuarterly($quarterly)
	{
		$this->data['quarterly'] = $quarterly;
	}


	/**
	 * Defines the KPIs yearly target
	 */
	public function getYearly()
	{
		return $this->data['yearly'];
	}
	public function setYearly($yearly)
	{
		$this->data['yearly'] = $yearly;
	}

	/**
	 * This represents the date from which these targets will be applied.
	 * This must be a UTC timestamp in ISO 8601 format with
	 * millisecond precision: YYYY-MM-DDTHH:MM:SS.MMMZ
	 */
	public function getStartDate()
	{
		return $this->data['startDate'];
	}
	public function setStartDate($startDate)
	{
		$this->data['startDate'] = $startDate;
	}


	/**
	 * This represents the date until which these targets will be applied.
	 * This must be a UTC timestamp in ISO 8601 format with
	 * millisecond precision: YYYY-MM-DDTHH:MM:SS.MMMZ
	 */
	public function getEndDate()
	{
		return $this->data['endDate'];
	}
	public function setEndDate($endDate)
	{
		$this->data['endDate'] = $endDate;
	}


	/**
	 * Defines the users who have personal targets by UUID
	 */
	public function getUserUUIDs()
	{
		return $this->data['userUUIDs'];
	}
	public function setUserUUIDs(array $userUUIDs)
	{
		$this->data['userUUIDs'] = $userUUIDs;
	}
	public function addUserUUIDs($uuid)
	{
		if (!isset($this->data['userUUIDs'])) {
			$this->data['userUUIDs'] = array();
		}
		array_push($this->data['userUUIDs'], (string) $uuid);
	}


	/**
	 * Defines the users who have personal targets
	 */
	public function getUserIds()
	{
		return $this->data['userIds'];
	}
	public function setUserIds(array $userIds)
	{
		$this->data['userIds'] = $userIds;
	}
	public function addUserIds($id)
	{
		if (!isset($this->data['userIds'])) {
			$this->data['userIds'] = array();
		}
		array_push($this->data['userIds'], (string) $id);
	}


	/**
	 * Defines the users by tag who have personal targets
	 */
	public function getTagIds()
	{
		return $this->data['tagIds'];
	}
	public function setTagIds(array $tagIds)
	{
		$this->data['tagIds'] = $tagIds;
	}
	public function addTagIds($id)
	{
		if (!isset($this->data['tagIds'])) {
			$this->data['tagIds'] = array();
		}
		array_push($this->data['tagIds'], (string) $id);
	}


	/**
	 * Defines the users by tag who have personal targets
	 */
	public function getTagTags()
	{
		return $this->data['tagTags'];
	}
	public function setTagTags(array $tagIds)
	{
		$this->data['tagTags'] = $tagIds;
	}
	public function addTagTags($tag)
	{
		if (!isset($this->data['tagTags'])) {
			$this->data['tagTags'] = array();
		}
		array_push($this->data['tagTags'], (string) $tag);
	}


	/**
	 * Return the JSON string equivalent of this object
	 */
	public function getJsonString(): string
	{
		return json_encode($this->data);
	}
}
