<?php
/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * This object captures the data required by the KPI API in
 * order to create / update KPIs.
 */

declare(strict_types=1);

namespace Mambo\Kpi\Data;

use Mambo\Common\Data\RequestData;
use Mambo\Common\Data\AbstractHasTagRequestData;

class KpiRequestData extends AbstractHasTagRequestData implements RequestData
{

	/**
	 * Whether the image associated to the KPI should be removed
	 * @return boolean removeImage
	 */
	public function getRemoveImage()
	{
		return $this->data['removeImage'];
	}
	public function setRemoveImage($removeImage)
	{
		$this->data['removeImage'] = $removeImage;
	}

	/**
	 * The name of the KPI
	 */
	public function getName()
	{
		return $this->data['name'];
	}
	public function setName($name)
	{
		$this->data['name'] = $name;
	}

	/**
	 * The description associated with the KPI
	 */
	public function getDescription()
	{
		return $this->data['description'];
	}
	public function setDescription($description)
	{
		$this->data['description'] = $description;
	}

	/**
	 * Indicates whether the KPI is active or not
	 */
	public function getActive()
	{
		return $this->data['active'];
	}
	public function setActive($active)
	{
		$this->data['active'] = $active;
	}

	/**
	 * Defines what mechanic is going to be tracked by the KPI
	 */
	public function getTracked()
	{
		return $this->data['tracked'];
	}
	public function setTracked(Tracked $tracked)
	{
		$this->data['tracked'] = $tracked;
	}

	/**
	 * Defines the Red, Amber, Green (RAG) status of the KPI
	 */
	public function getRedAmberGreen()
	{
		return $this->data['redAmberGreen'];
	}
	public function setRedAmberGreen(RedAmberGreen $redAmberGreen)
	{
		$this->data['redAmberGreen'] = $redAmberGreen;
	}

	/**
	 * Defines which users should have this KPI assigned to them by UUID
	 */
	public function getAssignToUserUUIDs()
	{
		return $this->data['assignToUserUUIDs'];
	}
	public function setAssignToUserUUIDs(array $assignToUserUUIDs)
	{
		$this->data['assignToUserUUIDs'] = $assignToUserUUIDs;
	}
	public function addAssignToUserUUIDs($assignToUserUUIDs)
	{
		if (!isset($this->data['assignToUserUUIDs'])) {
			$this->data['assignToUserUUIDs'] = array();
		}
		array_push($this->data['assignToUserUUIDs'], (string) $assignToUserUUIDs);
	}

	/**
	 * Defines which users should have this KPI assigned to them
	 */
	public function getAssignToUserIds()
	{
		return $this->data['assignToUserIds'];
	}
	public function setAssignToUserIds(array $assignToUserIds)
	{
		$this->data['assignToUserIds'] = $assignToUserIds;
	}
	public function addAssignToUserIds($assignToUserIds)
	{
		if (!isset($this->data['assignToUserIds'])) {
			$this->data['assignToUserIds'] = array();
		}
		array_push($this->data['assignToUserIds'], (string) $assignToUserIds);
	}

	/**
	 * Defines which users should have this KPI assigned to them
	 */
	public function getAssignToUsersByTagId()
	{
		return $this->data['assignToUsersByTagId'];
	}
	public function setAssignToUsersByTagId(array $assignToUsersByTagId)
	{
		$this->data['assignToUsersByTagId'] = $assignToUsersByTagId;
	}
	public function addAssignToUsersByTagId($assignToUsersByTagId)
	{
		if (!isset($this->data['assignToUsersByTagId'])) {
			$this->data['assignToUsersByTagId'] = array();
		}
		array_push($this->data['assignToUsersByTagId'], (string) $assignToUsersByTagId);
	}

	/**
	 * Defines if there are any suggested behaviours associated to this KPI.
	 * A suggested behaviour would be a behaviour that the users can complete
	 * in order to progress with the KPI.
	 */
	public function getSuggestedBehaviourIds()
	{
		return $this->data['suggestedBehaviourIds'];
	}
	public function setSuggestedBehaviourIds(array $suggestedBehaviourIds)
	{
		$this->data['suggestedBehaviourIds'] = $suggestedBehaviourIds;
	}
	public function addSuggestedBehaviourIds($suggestedBehaviourIds)
	{
		if (!isset($this->data['suggestedBehaviourIds'])) {
			$this->data['suggestedBehaviourIds'] = array();
		}
		array_push($this->data['suggestedBehaviourIds'], (string) $suggestedBehaviourIds);
	}

	/**
	 * Defines if there are any suggested behaviours by tag associated to this KPI.
	 * A suggested behaviour would be a behaviour that the users can complete
	 * in order to progress with the KPI.
	 */
	public function getSuggestedBehavioursByTagId()
	{
		return $this->data['suggestedBehavioursByTagId'];
	}
	public function setSuggestedBehavioursByTagId(array $suggestedBehavioursByTagId)
	{
		$this->data['suggestedBehavioursByTagId'] = $suggestedBehavioursByTagId;
	}
	public function addSuggestedBehavioursByTagId($suggestedBehavioursByTagId)
	{
		if (!isset($this->data['suggestedBehavioursByTagId'])) {
			$this->data['suggestedBehavioursByTagId'] = array();
		}
		array_push($this->data['suggestedBehavioursByTagId'], (string) $suggestedBehavioursByTagId);
	}

	/**
	 * Defines if there are any suggested rewards associated to this KPI.
	 * A suggested reward would be a reward that the users can complete
	 * in order to progress with the KPI.
	 */
	public function getSuggestedRewardIds()
	{
		return $this->data['suggestedRewardIds'];
	}
	public function setSuggestedRewardIds(array $suggestedRewardIds)
	{
		$this->data['suggestedRewardIds'] = $suggestedRewardIds;
	}
	public function addSuggestedRewardIds($suggestedRewardIds)
	{
		if (!isset($this->data['suggestedRewardIds'])) {
			$this->data['suggestedRewardIds'] = array();
		}
		array_push($this->data['suggestedRewardIds'], (string) $suggestedRewardIds);
	}

	/**
	 * Defines if there are any suggested rewards by tag associated to this KPI.
	 * A suggested reward would be a reward that the users can complete
	 * in order to progress with the KPI.
	 */
	public function getSuggestedRewardsByTagId()
	{
		return $this->data['suggestedRewardsByTagId'];
	}
	public function setSuggestedRewardsByTagId(array $suggestedRewardsByTagId)
	{
		$this->data['suggestedRewardsByTagId'] = $suggestedRewardsByTagId;
	}
	public function addSuggestedRewardsByTagId($suggestedRewardsByTagId)
	{
		if (!isset($this->data['suggestedRewardsByTagId'])) {
			$this->data['suggestedRewardsByTagId'] = array();
		}
		array_push($this->data['suggestedRewardsByTagId'], (string) $suggestedRewardsByTagId);
	}

	/**
	 * Indicates whether the KPI will distribute prizes when targets are met.
	 */
	public function getHasPrizes()
	{
		return $this->data['hasPrizes'];
	}
	public function setHasPrizes($hasPrizes)
	{
		$this->data['hasPrizes'] = $hasPrizes;
	}

	/**
	 * The prizes object is used to define what prizes will be unlocked
	 * when the user hits this KPIs targets. Note that the way the points
	 * are distributed can be modified using the prizePointBonus and
	 * pointsDistributionType properties.
	 */
	public function getPrizes()
	{
		return $this->data['prizes'];
	}
	public function setPrizes(KpiPrize $prizes)
	{
		$this->data['prizes'] = $prizes;
	}

	/**
	 * Custom fields defined for the KPI. These can contain additional
	 * data or any kind of information you would like to store which isn't a
	 * standard field of the KPI.
	 */
	public function getCustomFields()
	{
		return $this->data['customFields'];
	}
	public function setCustomFields(array $customFields)
	{
		$this->data['customFields'] = $customFields;
	}

	/**
	 * This contains the list of the translations which must be used with the object.
	 * Setting a property directly on the object (such as using setName()) will add
	 * the property to the list of translations using the default language of the server.
	 * If you set a property on the object directly (such as setName()) and then set
	 * the same property using setTranslations(), then the setTranslations() property
	 * will take priority.
	 */
	public function getTranslations()
	{
		return $this->data['translations'];
	}
	public function setTranslations(array $translations)
	{
		$this->data['translations'] = $translations;
	}
	public function addTranslation(KpiTranslation $translation)
	{
		if (!isset($this->data['translations'])) {
			$this->data['translations'] = array();
		}
		array_push($this->data['translations'], $translation);
	}


	/**
	 * Return the JSON string equivalent of this object
	 */
	public function getJsonString(): string
	{
		$json = $this->data;

		if (isset($json['prizes'])) {
			$json['prizes'] = $json['prizes']->getJsonArray();
		}

		if (isset($json['tracked'])) {
			$json['tracked'] = $json['tracked']->getJsonArray();
		}

		if (isset($json['redAmberGreen'])) {
			$json['redAmberGreen'] = $json['redAmberGreen']->getJsonArray();
		}

		if (isset($json['bonusPoints'])) {
			$bonusPointsArr = array();
			foreach ($json['bonusPoints'] as $bonusPoints) {
				array_push($bonusPointsArr, $bonusPoints->getJsonArray());
			}
			$json['bonusPoints'] = $bonusPointsArr;
		}

		if (isset($json['customFields'])) {
			$customFieldsArr = array();
			foreach ($json['customFields'] as $customField) {
				array_push($customFieldsArr, $customField->getJsonArray());
			}
			$json['customFields'] = $customFieldsArr;
		}

		if (isset($json['translations'])) {
			$translationsArr = array();
			foreach ($json['translations'] as $translation) {
				array_push($translationsArr, $translation->getJsonArray());
			}
			$json['translations'] = $translationsArr;
		}

		return json_encode($json);
	}
}
