<?php
/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * This class encapsulates data related to a KPIs prizes
 */

declare(strict_types=1);

namespace Mambo\Kpi\Data;

use Mambo\Common\Data\Prize;

class KpiPrize
{
    private $data = array();

    /**
     * This determines for which of the target time periods the prize
     * points will be given. This includes both the regular prize points
     * and the bonus points (if any). The default value is Monthly.
     */
    public function getPrizeTargetPeriod()
    {
        return $this->data['prizeTargetPeriod'];
    }
    public function setPrizeTargetPeriod($prizeTargetPeriod)
    {
        $this->data['prizeTargetPeriod'] = $prizeTargetPeriod;
    }

    /**
     * This determines how the KPI's prize points will be distributed.
     * Options include: (1) on_complete: this means the points will be
     * distributed when the KPI target is hit; (2) on_progress: this means
     * the points will be distributed based on the progress made towards
     * achieving the KPI (e.g. 80% progress means 80% of the points will
     * be distributed)
     */
    public function getPointsDistributionType()
    {
        return $this->data['pointsDistributionType'];
    }
    public function setPointsDistributionType($pointsDistributionType)
    {
        $this->data['pointsDistributionType'] = $pointsDistributionType;
    }

    /**
     * This determines whether bonus points will be distributed to those
     * who achieve more than 100% of the target. The options for the bonus
     * include: (1) none: no bonus points; (2) percentage: the percentage
     * over the target is translated into a percentage of the total prize
     * points available (e.g. 1000 total points available, user hits 120%
     * of the target, we take 20% of the 1000 points and give them to the
     * user i.e. an extra 200 points)
     */
    public function getBonusPointsType()
    {
        return $this->data['bonusPointsType'];
    }
    public function setBonusPointsType($bonusPointsType)
    {
        $this->data['bonusPointsType'] = $bonusPointsType;
    }

    /**
     * The prizes object is used to define what prizes will be unlocked
     * when the user hits this KPIs targets. Note that the way the points
     * are distributed can be modified using the prizePointBonus and
     * pointsDistributionType properties.
     */
    public function getPrizes()
    {
        return $this->data['prizes'];
    }
    public function setPrizes(Prize $prizes)
    {
        $this->data['prizes'] = $prizes;
    }

    /**
     * Returns the current model as an array ready to
     * be json_encoded
     */
    public function getJsonArray()
    {
        $json = $this->data;

        if (isset($json['prizes'])) {
            $json['prizes'] = $json['prizes']->getJsonArray();
        }

        return $json;
    }
}
