<?php
/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

declare(strict_types=1);

namespace Mambo\Http;

use Mambo\ClientConfiguration;

class RequestOptions
{
    private ?string $idempotencyKey = null;
    private ?string $acceptLanguage = null;
    private ?int $timeoutMilliseconds = null;
    private ?int $connectTimeoutMilliseconds = null;

    /**
     * The IdempotencyKey to be used for the request
     */
    public function getIdempotencyKey(): ?string
    {
        return $this->idempotencyKey;
    }
    public function setIdempotencyKey(string $idempotencyKey): self
    {
        $this->idempotencyKey = $idempotencyKey;
        return $this;
    }

    /**
     * The accept language header value used for the request.
     * Currently supported values are: en / pt
     */
    public function getAcceptLanguage(): ?string
    {
        return $this->acceptLanguage;
    }
    public function setAcceptLanguage(string $acceptLanguage): self
    {
        $this->acceptLanguage = $acceptLanguage;
        return $this;
    }

    /**
     * Set the timeout in milliseconds to read a response from the server.
     * The default value is 60.
     * A value of 0 represents infinity.
     */
    public function getTimeoutMilliseconds(): ?int
    {
        return $this->timeoutMilliseconds;
    }
    public function setTimeoutMilliseconds(int $timeout): self
    {
        $this->timeoutMilliseconds = $timeout;
        return $this;
    }

    /**
     * Set the timeout in milliseconds to connect to the server.
     * The default value is 20.
     * A value of 0 is interpreted as 300 seconds.
     */
    public function getConnectTimeoutMilliseconds(): ?int
    {
        return $this->connectTimeoutMilliseconds;
    }
    public function setConnectTimeoutMilliseconds(int $timeout): self
    {
        $this->connectTimeoutMilliseconds = $timeout;
        return $this;
    }

    /**
     * Merges configuration values from ClientConfiguration into this RequestOptions
     * where values are not already set.
     */
    public function merge(ClientConfiguration $config): self
    {
        if ($this->acceptLanguage === null) {
            $this->acceptLanguage = $config->getAcceptLanguage();
        }

        if ($this->timeoutMilliseconds === null) {
            $this->timeoutMilliseconds = $config->getTimeoutMilliseconds();
        }

        if ($this->connectTimeoutMilliseconds === null) {
            $this->connectTimeoutMilliseconds = $config->getConnectTimeoutMilliseconds();
        }

        return $this;
    }
}
