<?php
/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * Fires the requests to the Mambo API. This class wraps the CURL
 * client with helper functions used to decode the responses from
 * the Mambo API as well as handle any Exceptions that may be returned.
 */

declare(strict_types=1);

namespace Mambo\Http;

use Mambo\Exception\MamboExceptionFactory;
use Mambo\Exception\ApiConnectionException;
use Mambo\Exception\OAuth\InvalidTokenException;
use Mambo\Common\Data\RequestData;
use Mambo\Util\Preconditions;
use Mambo\Json\JsonMapper;
use Mambo\ClientConfiguration;

class HttpClient
{
    /**
     * The PHP client version
     * @var string
     */
    const API_CLIENT_VERSION = '8.7.2';

    /**
     * The default encoding used by the Client
     * @var string
     */
    const DEFAULT_ENCODING = 'UTF-8';

    /**
     * HTTP method types
     * @var string
     */
    const GET = 'GET';
    const POST = 'POST';
    const PUT = 'PUT';
    const DELETE = 'DELETE';

    private $userAgent = 'User-Agent: Mambo/' . self::API_CLIENT_VERSION . '; PHP ' . PHP_VERSION;

    /**
     * API URL appended to the serverUrl
     * @var string
     */
    private $apiUrlPath = "/api";

    /**
     * The base URL of the API
     * @var string
     */
    private $serverUrl = null;

    /**
     * The URL used to request an access token
     * @var string
     */
    private $accessTokenUrl = "/oauth/token";

    /**
     * The access token and refresh token to be used for API calls
     */
    private $oauthToken = null;
    private $refreshTokenAttempts = 0;


    private JsonMapper $jsonMapper;
    private MamboExceptionFactory $exceptionFactory;
    private MamboCredentials $credentials;
    private ClientConfiguration $configuration;


    /**
     * Constructor
     */
    public function __construct(MamboCredentials $credentials, ClientConfiguration $configuration)
    {
        $this->jsonMapper = new JsonMapper();
        $this->exceptionFactory = new MamboExceptionFactory();
        $this->credentials = $credentials;
        $this->configuration = $configuration;

        $this->updateEndPointBaseUrl();
    }


    /**
     * Get the version of the SDK
     */
    public function getVersion(): string
    {
        return self::API_CLIENT_VERSION;
    }


    /**
     * Updates the base portion of the URL.
     * The default value is: https://api.mambo.io/api
     */
    private function updateEndPointBaseUrl()
    {
        Preconditions::checkNotNull(
            $this->configuration->getServerBaseUrl(),
            "serverBaseUrl must be set in the ClientConfiguration"
        );

        $this->serverUrl = $this->configuration->getServerBaseUrl() . $this->apiUrlPath;
    }


    /**
     * Upload an Image to the Mambo.IO platform
     *
     * @param ApiUrlBuilder $urlBuilder         The URL builder that generates the relevant URL
     * @param string|false $image               The image to be sent with the request
     * @param RequestData $data                 The data to be sent with the request
     * @param RequestOptions $requestOptions    The requestOptions to be used with the request
     */
    public function upload(ApiUrlBuilder $urlBuilder, string|false $image, ?RequestData $data = null, ?RequestOptions $requestOptions = null): mixed
    {
        $this->refreshTokenAttempts = 0;
        return $this->doRequest($urlBuilder, HttpClient::POST, $data, $image, $requestOptions);
    }


    /**
     * Send an HTTP request to the Mambo API
     *
     * @param ApiUrlBuilder $urlBuilder         The URL builder that generates the relevant URL
     * @param string $method					Specifies the HTTP method to be used for this request
     * @param RequestData $data					The data to be sent with the request
     * @param RequestOptions $requestOptions	The requestOptions to be used with the request
     */
    public function request(ApiUrlBuilder $urlBuilder, string $method, ?RequestData $data = null, ?RequestOptions $requestOptions = null): mixed
    {
        $this->refreshTokenAttempts = 0;
        return $this->doRequest($urlBuilder, $method, $data, false, $requestOptions);
    }


    /**
     * Send an HTTP request to the Mambo API
     *
     * @param ApiUrlBuilder $urlBuilder         The URL builder that generates the relevant URL
     * @param string $method					Specifies the HTTP method to be used for this request
     * @param RequestData $data					The data to be sent with the request
     * @param string|false $image				The image to be sent with the request
     * @param RequestOptions $requestOptions	The requestOptions to be used with the request
     */
    private function doRequest(
        ApiUrlBuilder $urlBuilder,
        string $method,
        ?RequestData $data = null,
        string|false $image = null,
        ?RequestOptions $requestOptions = null
    ): mixed {
        // Set the default encoding
        if (function_exists('mb_internal_encoding')) {
            mb_internal_encoding(self::DEFAULT_ENCODING);
        }

        // Get the fully qualified URL
        $urlBuilder->withServerUrl($this->serverUrl);

        // Debugging
        if ($this->configuration->getDebug()) {
            echo "------------------------\n";
            echo "|-Method: " . $method . "\n";
            echo "|-URL: " . $urlBuilder->build() . "\n";
            echo "|-Request Data: " . ($data !== null ? $data->getJsonString() : 'null') . "\n";
            echo "|-Request Options: " . json_encode($requestOptions) . "\n";
        }

        $requestOptions = $this->getOrDefault($requestOptions, new RequestOptions());
        $requestOptions->merge($this->configuration);

        $options = $this->getCurlOptions($urlBuilder, $method, $data, $image, $requestOptions);

        try {
            $response = $this->execRequest($options);
        } catch (InvalidTokenException $ex) {
            return $this->refreshToken($ex, $urlBuilder, $method, $data, $image, $requestOptions);
        }

        return $response;
    }


    /**
     * @template T
     * @param T|null $value
     * @param T $default
     * @return T
     */
    private function getOrDefault($value, $default)
    {
        return $value !== null ? $value : $default;
    }


    /**
     * Initialise all the cURL options required to execute the request
     *
     * @param ApiUrlBuilder $urlBuilder         The URL builder that generates the relevant URL
     * @param string $method					Specifies the HTTP method to be used for this request
     * @param RequestData $data					The data to be sent with the request
     * @param string|false $image				The image to be sent with the request
     * @param RequestOptions $requestOptions	The requestOptions to be used with the request
     * @return array
     */
    private function getCurlOptions(ApiUrlBuilder $urlBuilder, string $method, ?RequestData $data, string|false $image, RequestOptions $requestOptions): array
    {
        $this->ensureValidAccessToken();

        $options = [
            CURLOPT_HTTPHEADER => $this->buildHeaders($image, $requestOptions),
            CURLOPT_HEADER => false,
            CURLOPT_ENCODING => '',
            CURLOPT_URL => $urlBuilder->build(),
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_SSL_VERIFYHOST => false,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_FAILONERROR => false,
            CURLOPT_MAXREDIRS => 1,
            CURLOPT_CONNECTTIMEOUT => $requestOptions->getConnectTimeoutMilliseconds(),
            CURLOPT_TIMEOUT => $requestOptions->getTimeoutMilliseconds(),
            CURLOPT_USERAGENT => $this->userAgent
        ];

        if ($method === self::POST) {
            $options[CURLOPT_POST] = true;
            $options[CURLOPT_POSTFIELDS] = $this->buildPostFields($data, $image);
        } elseif ($method === self::PUT) {
            $options[CURLOPT_CUSTOMREQUEST] = 'PUT';
            $options[CURLOPT_POSTFIELDS] = $data->getJsonString();
        } elseif ($method === self::DELETE) {
            $options[CURLOPT_CUSTOMREQUEST] = $method;
            if ($data !== null) {
                $options[CURLOPT_POSTFIELDS] = $data->getJsonString();
            }
        }

        return $options;
    }


    private function ensureValidAccessToken(): void
    {
        if (is_null($this->oauthToken)) {
            $this->updateAccessTokenWithCredentials();
        }
    }


    private function buildHeaders(string|false $image, RequestOptions $requestOptions): array
    {
        $contentType = empty($image) ? 'application/json' : 'multipart/form-data';

        $headers = array(
            'Content-Type: ' . $contentType . '; charset=' . self::DEFAULT_ENCODING,
            'Accept: application/json',
            'Accept-Language: ' . $requestOptions->getAcceptLanguage(),
            'Accept-Version: ' . $this->getVersion(),
            'Authorization: Bearer ' . $this->oauthToken
        );

        if (!empty($requestOptions->getIdempotencyKey())) {
            array_push($headers, 'Idempotency-Key: ' . $requestOptions->getIdempotencyKey());
        }

        return $headers;
    }


    private function buildPostFields(?RequestData $data, string|false $image)
    {
        if (empty($image)) {
            return (empty($data)) ? $data : $data->getJsonString();
        }

        return array_filter(array(
            'image' => new \CURLFile($image),
            'data' => (empty($data)) ? $data : $data->getJsonString()
        ));
    }


    /**
     * Updates the access token required to make API calls
     */
    private function updateAccessTokenWithCredentials(): void
    {
        $header = array(
            'application/x-www-form-urlencoded;charset=utf-8',
            'Accept: application/json',
            'Accept-Language: ' . $this->configuration->getAcceptLanguage()
        );

        $options = array(
            CURLOPT_HTTPHEADER => $header,
            CURLOPT_HEADER => false,
            CURLOPT_ENCODING => '',
            CURLOPT_URL => $this->configuration->getServerBaseUrl() . $this->accessTokenUrl,
            CURLOPT_USERPWD => $this->credentials->getPublicKey() . ":" . $this->credentials->getPrivateKey(),
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => 'grant_type=client_credentials',
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_SSL_VERIFYHOST => false,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_FAILONERROR => false,
            CURLOPT_MAXREDIRS => 1,
            CURLOPT_CONNECTTIMEOUT_MS => $this->configuration->getConnectTimeoutMilliseconds(),
            CURLOPT_TIMEOUT_MS => $this->configuration->getTimeoutMilliseconds(),
            CURLOPT_USERAGENT => $this->userAgent
        );

        $response = $this->execRequest($options);
        $this->oauthToken = ((array)$response)['access_token'];
    }


    /**
     * Executes the cURL request to the API
     *
     * @param array $options	The cURL options used to make the request
     * @return string
     */
    private function execRequest(array $options): mixed
    {
        // Initialise and execute the request
        $conn = curl_init();
        curl_setopt_array($conn, $options);
        $json = curl_exec($conn);

        if ($json === false) {
            $error = curl_error($conn);
            curl_close($conn);
            throw new ApiConnectionException("Curl request failed: " . $error);
        }

        $httpStatus = curl_getinfo($conn, CURLINFO_HTTP_CODE);
        curl_close($conn);

        // Debugging
        if ($this->configuration->getDebug()) {
            echo "|-Response: " . $json . "\n";
        }

        // Decode the json response
        $response = $this->jsonMapper->parse($json, $this->configuration->getJsonDecodeAsArray());

        // Throw any errors
        if ($this->hasError($response)) {
            throw $this->exceptionFactory->createMamboApiException(
                $json,
                $httpStatus
            );
        }

        return $response;
    }


    /**
     * Returns true if the server response contains an error
     */
    private function hasError($response): bool
    {
        return isset(((array)$response)['error']);
    }


    /**
     * Refreshes the token if we haven't already tried a few times
     */
    private function refreshToken(
        InvalidTokenException $exception,
        ApiUrlBuilder $urlBuilder,
        string $method,
        $data = null,
        $image = null,
        RequestOptions $requestOptions = null
    ): mixed {
        if ($this->refreshTokenAttempts > 5) {
            throw $exception;
        }

        $this->refreshTokenAttempts++;
        $this->oauthToken = null;
        return $this->doRequest($urlBuilder, $method, $data, $image, $requestOptions);
    }
}
