<?php
/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * Responsible for building and validating the API request URL
 */

declare(strict_types=1);

namespace Mambo\Http;

use InvalidArgumentException;
use Mambo\Common\Param\ServiceParams;
use Mambo\Util\Preconditions;

class ApiUrlBuilder
{
	/** @var string */
	private $serverUrl;

	/** @var string */
	private $urlTemplate;

	/** @var array<string, string> */
	private $pathParams = [];

	/** @var ServiceParams|null */
	private $queryParams;

	public function __construct(string $urlTemplate)
	{
		Preconditions::checkStartsWith($urlTemplate, "/", "urlTemplate must start with /");
		$this->urlTemplate = $urlTemplate;
	}

	/**
	 * The base URL that will be prepended to the generated URL
	 * @param string $serverUrl
	 * @return self
	 */
	public function withServerUrl(string $serverUrl): self
	{
		$this->serverUrl = $serverUrl;
		return $this;
	}

	/**
	 * Add a custom path parameter
	 * @param string $key The key for the path parameter
	 * @param string|int $value The value to substitute
	 * @return self
	 */
	public function withPathParam(string $key, $value): self
	{
		$this->pathParams[$key] = (string)$value;
		return $this;
	}

	/**
	 * Add site path parameter
	 * @param string $siteUrl The site URL to substitute into the path
	 * @return self
	 */
	public function withPathSite(string $siteUrl): self
	{
		$this->pathParams['site'] = $siteUrl;
		return $this;
	}

	/**
	 * Add uuid path parameter
	 * @param string $uuid The UUID to substitute into the path
	 * @return self
	 */
	public function withPathUuid(string $uuid): self
	{
		$this->pathParams['uuid'] = $uuid;
		return $this;
	}

	/**
	 * Add id path parameter 
	 * @param string $id The ID to substitute into the path
	 * @return self
	 */
	public function withPathId(string $id): self
	{
		$this->pathParams['id'] = $id;
		return $this;
	}

	public function withQueryParams(?ServiceParams $params): self
	{
		$this->queryParams = $params;
		return $this;
	}

	public function build(): string
	{
		Preconditions::checkNotEmpty($this->serverUrl, "serverUrl must not be empty");

		$apiUrl = $this->buildBaseUrl();
		$queryString = $this->buildQueryString();

		return $this->serverUrl . $apiUrl . $queryString;
	}

	private function buildBaseUrl(): string
	{
		$requiredParams = $this->getRequiredParams();
		$this->validateParams($requiredParams);

		$url = $this->urlTemplate;
		foreach ($this->pathParams as $key => $value) {
			if (!in_array($key, $requiredParams, true)) {
				throw new InvalidArgumentException(sprintf('Unexpected path parameter provided: %s', $key));
			}
			$url = str_replace(sprintf("{%s}", $key), $value, $url);
		}

		return $url;
	}

	/**
	 * @return array<string>
	 */
	private function getRequiredParams(): array
	{
		$matches = [];
		preg_match_all('/{([^}]+)}/', $this->urlTemplate, $matches);
		return $matches[1];
	}

	/**
	 * @param array<string> $requiredParams
	 * @throws InvalidArgumentException
	 */
	private function validateParams(array $requiredParams): void
	{
		foreach ($requiredParams as $param) {
			if (!isset($this->pathParams[$param])) {
				throw new InvalidArgumentException(sprintf('Missing required path parameter: %s', $param));
			}
		}
	}

	private function buildQueryString(): string
	{
		if ($this->queryParams === null) {
			return '';
		}

		$queryParts = [];
		foreach ($this->queryParams->asMap() as $key => $values) {
			foreach ($values as $value) {
				$queryParts[] = sprintf(
					'%s=%s',
					urlencode($key),
					urlencode((string)$value)
				);
			}
		}

		if (empty($queryParts)) {
			return '';
		}

		return '?' . implode('&', $queryParts);
	}
}
