<?php
/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

declare(strict_types=1);

namespace Mambo\Exception;

use Mambo\Exception\OAuth\OAuthError;
use Mambo\Exception\OAuth\OAuthException;
use Mambo\Exception\OAuth\InvalidRequestException;
use Mambo\Exception\OAuth\InvalidTokenException;
use Mambo\Exception\OAuth\UnsupportedGrantTypeException;
use Mambo\Json\JsonMapper;

class MamboExceptionFactory
{
    private static JsonMapper $jsonMapper;

    public function __construct()
    {
        self::$jsonMapper = new JsonMapper();
    }

    public function createMamboApiException(
        string $jsonString,
        int $status
    ): MamboApiException {
        $response = self::$jsonMapper->parse($jsonString, true);
        $type = isset($response['error']['type']) ? $response['error']['type'] : $response['type'];

        switch ($type) {
            case 'InvalidIdempotencyKeyException':
            case 'IdempotencyKeyAlreadyUsedException':
            case 'IdempotencyRequestInProgressException':
                return new IdempotencyException($this->buildMamboError($response), $status);
            case 'UnauthorizedException':
            case 'ForbiddenException':
                return new AuthenticationException($this->buildMamboError($response), $status);
            case 'OAuth2Exception':
                return $this->createOAuthException($response, $status);
            default:
                return new MamboApiException($this->buildMamboError($response), $status);
        }
    }


    private function buildMamboError(array $response): MamboError
    {
        return new MamboError($response);
    }


    private function createOAuthException(array $response, int $status): OAuthException
    {
        $error = new OAuthError($response);

        switch ($error->getError()) {
            case 'invalid_token':
                return new InvalidTokenException($error, $status);
            case 'invalid_request':
                return new InvalidRequestException($error, $status);
            case 'unsupported_grant_type':
                return new UnsupportedGrantTypeException($error, $status);
            default:
                return new OAuthException($error, $status);
        }
    }
}
