<?php
/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

declare(strict_types=1);

namespace Mambo\DataStore\Param;

use Mambo\Common\Param\AbstractServiceParams;

/**
 * Parameters for retrieving a list of data stores
 * @extends AbstractServiceParams<DataStoreGetListParams>
 */
class DataStoreGetListParams extends AbstractServiceParams
{

    /**
     * @param int $page Specifies the page of results to retrieve
     * @return self
     */
    public function page(int $page): self
    {
        return $this->extraParams('page', $page);
    }

    /**
     * @param int $count Specifies the number of results to retrieve, up to a maximum of 100
     * @return self
     */
    public function count(int $count): self
    {
        return $this->extraParams('count', $count);
    }

    /**
     * @param string $type The type of data store to retrieve
     * @return self
     */
    public function type(string $type): self
    {
        return $this->extraParams('type', $type);
    }

    /**
     * @param array<string> $ownedByIds The ownedBy IDs by which to filter the data stores retrieved
     * @return self
     */
    public function ownedByIds(array $ownedByIds): self
    {
        return $this->extraParamsList('ownedByIds', $ownedByIds);
    }

    /**
     * @param array<string> $ownedByUuids The ownedBy UUIDs by which to filter the data stores retrieved
     * @return self
     */
    public function ownedByUuids(array $ownedByUuids): self
    {
        return $this->extraParamsList('ownedByUuids', $ownedByUuids);
    }

    /**
     * @param array<string> $userIds The user IDs by which to filter the data stores retrieved
     * @return self
     */
    public function userIds(array $userIds): self
    {
        return $this->extraParamsList('userIds', $userIds);
    }

    /**
     * @param array<string> $userUuids The user UUIDs by which to filter the data stores retrieved
     * @return self
     */
    public function userUuids(array $userUuids): self
    {
        return $this->extraParamsList('userUuids', $userUuids);
    }

    /**
     * @param string $startDate Specifies the start date of the data stores retrieved
     * @return self
     */
    public function startDate(string $startDate): self
    {
        return $this->extraParams('startDate', $startDate);
    }

    /**
     * @param string $endDate Specifies the end date of the data stores retrieved
     * @return self
     */
    public function endDate(string $endDate): self
    {
        return $this->extraParams('endDate', $endDate);
    }

    /**
     * @param string $order Specifies the order in which to sort the results
     * @return self
     */
    public function order(string $order): self
    {
        return $this->extraParams('order', $order);
    }

    /**
     * @param bool $includeSystemDataStores Specifies whether to include system data stores
     * @return self
     */
    public function includeSystemDataStores(bool $includeSystemDataStores): self
    {
        return $this->extraParams('includeSystemDataStores', $includeSystemDataStores);
    }

    /**
     * @param bool $withUsers Specifies whether to return data stores with full user objects
     * @return self
     */
    public function withUsers(bool $withUsers): self
    {
        return $this->extraParams('withUsers', $withUsers);
    }

    /**
     * @param array<string> $userProperties The list of user properties to be included
     * @return self
     */
    public function userProperties(array $userProperties): self
    {
        return $this->extraParamsList('userProperties', $userProperties);
    }
}
