<?php
/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * The MamboDataStoresService class handles all DataStore related requests
 * to the Mambo API.
 */

declare(strict_types=1);

namespace Mambo\DataStore;

use Mambo\DataStore\Data\DataStoreRequestData;
use Mambo\DataStore\Param\DataStoreGetParams;
use Mambo\DataStore\Param\DataStoreGetListParams;
use Mambo\DataStore\Param\DataStoreDeleteByCriteriaParams;
use Mambo\Common\Service\AbstractService;
use Mambo\Common\Data\DeleteRequestData;
use Mambo\Http\ApiUrlBuilder;
use Mambo\Http\RequestOptions;

class DataStoresService extends AbstractService
{
	/**
	 * DataStore service end data store URI
	 * @var string
	 */
	const DATA_STORES_URI = "/v1/data_stores";
	const DATA_STORES_ID_URI = "/v1/data_stores/{id}";
	const DATA_STORES_CLONE_URI = "/v1/data_stores/{id}/clone";

	const DATA_STORES_SITE_URI = "/v1/{site}/data_stores";


	/**
	 * This method is used to create a new data store.
	 *
	 * @param string $siteUrl					The site to which the data store belongs to
	 * @param DataStoreRequestData $data			The data sent to the API in order to create a data store
	 * @param RequestOptions $requestOptions		The requestOptions to be used with the request
	 */
	public function create(string $siteUrl, DataStoreRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::DATA_STORES_SITE_URI);
		$urlBuilder->withPathSite($siteUrl);
		return $this->httpPost($urlBuilder, $requestOptions, $data);
	}


	/**
	 * Update an existing data store by ID.
	 *
	 * @param string $id					The ID of the data store to update
	 * @param DataStoreRequestData $data	The data with which to update the specified data store object
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function update(string $id, DataStoreRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::DATA_STORES_ID_URI);
		$urlBuilder->withPathId($id);
		return $this->httpPut($urlBuilder, $requestOptions, $data);
	}


	/**
	 * Clone a data store
	 *
	 * @param string $id		The ID of the data store to clone
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function cloneDataStore(string $id, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::DATA_STORES_CLONE_URI);
		$urlBuilder->withPathId($id);
		return $this->httpPost($urlBuilder, $requestOptions);
	}


	/**
	 * Delete a data store by it's ID
	 *
	 * @param string $id				The ID of the data store to delete
	 * @param boolean $includeSystemDataStores  Whether system data stores, if any, should be included in the delete
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function delete(string $id, $includeSystemDataStores = null, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::DATA_STORES_ID_URI);
		$urlBuilder->withPathId($id);

		return $this->httpDelete($urlBuilder, $requestOptions);
	}


	/**
	 * Delete a list of data stores by their ID
	 *
	 * @param DeleteRequestData $data			The list of IDs of the data store to delete
	 * @param boolean $includeSystemDataStores  Whether system data stores, if any, should be included in the delete
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function deleteDataStores(DeleteRequestData $data, $includeSystemDataStores = null, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::DATA_STORES_URI);

		return $this->httpDelete($urlBuilder, $requestOptions, $data);
	}


	/**
	 * Delete a list of data stores by the criteria specified
	 *
	 * @param string $siteUrl							The site for which to retrieve the list of data stores
	 * @param DataStoreDeleteByCriteriaParams $params	The query parameters to be used with the request
	 * @param RequestOptions $requestOptions			The requestOptions to be used with the request
	 */
	public function deleteDataStoresByCriteria(
		string $siteUrl,
		DataStoreDeleteByCriteriaParams $params = null,
		RequestOptions $requestOptions = null
	): array|object {

		$urlBuilder = new ApiUrlBuilder(self::DATA_STORES_SITE_URI);
		$urlBuilder->withPathSite($siteUrl);
		$urlBuilder->withQueryParams($params);

		return $this->httpDelete($urlBuilder, $requestOptions);
	}


	/**
	 * Get a data store by it's ID
	 *
	 * @param string $id						The ID of the data store to retrieve
	 * @param DataStoreGetParams $params		The query parameters to be used with the request
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function get(string $id, DataStoreGetParams $params = null, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::DATA_STORES_ID_URI);
		$urlBuilder->withPathId($id);
		$urlBuilder->withQueryParams($params);

		return $this->httpGet($urlBuilder, $requestOptions);
	}


	/**
	 * Get the list of data stores for the specified site
	 *
	 * @param string $siteUrl					The site for which to retrieve the list of data stores
	 * @param DataStoreGetListParams $params	The query parameters to be used with the request
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function getDataStores(
		string $siteUrl,
		DataStoreGetListParams $params = null,
		RequestOptions $requestOptions = null
	): array|object {

		$urlBuilder = new ApiUrlBuilder(self::DATA_STORES_SITE_URI);
		$urlBuilder->withPathSite($siteUrl);
		$urlBuilder->withQueryParams($params);

		return $this->httpGet($urlBuilder, $requestOptions);
	}
}
