<?php
/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * This object captures the data required by the DataStore API in
 * order to create / update data stores.
 */

declare(strict_types=1);

namespace Mambo\DataStore\Data;

use Mambo\Common\Data\RequestData;

class DataStoreRequestData implements RequestData
{
	private $data = array();


	/**
	 * The name of the data store.
	 */
	public function getName()
	{
		return $this->data['name'];
	}
	public function setName($name)
	{
		$this->data['name'] = $name;
	}

	/**
	 * The type of the data store.
	 */
	public function getType()
	{
		return $this->data['type'];
	}
	public function setType($type)
	{
		$this->data['type'] = $type;
	}

	/**
	 * The data of the data store.
	 */
	public function getData()
	{
		return $this->data['data'];
	}
	public function setData($data)
	{
		$this->data['data'] = $data;
	}

	/**
	 * The AggregateId of the user who owns this data store.
	 */
	public function getOwnedBy()
	{
		return $this->data['ownedBy'];
	}
	public function setOwnedBy($ownedBy)
	{
		$this->data['ownedBy'] = $ownedBy;
	}

	/**
	 * The UUID of the user who owns this data store.
	 */
	public function getOwnedByUuid()
	{
		return $this->data['ownedByUuid'];
	}
	public function setOwnedByUuid($ownedByUuid)
	{
		$this->data['ownedByUuid'] = $ownedByUuid;
	}

	/**
	 * Any user IDs associated to this data store by UUID
	 */
	public function getUserUUIDs()
	{
		return $this->data['userUUIDs'];
	}
	public function setUserUUIDs(array $userUUIDs)
	{
		$this->data['userUUIDs'] = $userUUIDs;
	}
	public function addUserUUIDs($uuid)
	{
		if (!isset($this->data['userUUIDs'])) {
			$this->data['userUUIDs'] = array();
		}
		array_push($this->data['userUUIDs'], (string) $uuid);
	}

	/**
	 * Any user IDs associated to this data store.
	 */
	public function getUserIds()
	{
		return $this->data['userIds'];
	}
	public function setUserIds(array $userIds)
	{
		$this->data['userIds'] = $userIds;
	}
	public function addUserIds($id)
	{
		if (!isset($this->data['userIds'])) {
			$this->data['userIds'] = array();
		}
		array_push($this->data['userIds'], (string) $id);
	}

	/**
	 * Any time stamp associated to the data store.
	 * Allows you to filter the data stores by date.
	 * This must be a UTC timestamp in ISO 8601 format with
	 * millisecond precision: YYYY-MM-DDTHH:MM:SS.MMMZ
	 */
	public function getTimestamp()
	{
		return $this->data['timestamp'];
	}
	public function setTimestamp($timestamp)
	{
		$this->data['timestamp'] = $timestamp;
	}


	/**
	 * Returns the current model as an array ready to
	 * be json_encoded
	 */
	public function getJsonString(): string
	{
		return json_encode($this->data);
	}
}
