<?php
/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * The MamboCustomFieldsService class handles all CustomField related requests
 * to the Mambo API.
 */

declare(strict_types=1);

namespace Mambo\CustomField;

use Mambo\CustomField\Data\CustomFieldRequestData;
use Mambo\Common\Service\AbstractService;
use Mambo\Common\Data\DeleteRequestData;
use Mambo\Http\ApiUrlBuilder;
use Mambo\Http\RequestOptions;

class CustomFieldsService extends AbstractService
{
	/**
	 * CustomField service end custom field URI
	 * @var string
	 */
	const CUSTOM_FIELDS_URI = "/v1/custom_fields";
	const CUSTOM_FIELDS_ID_URI = "/v1/custom_fields/{id}";
	const CUSTOM_FIELDS_CLONE_URI = "/v1/custom_fields/{id}/clone";

	const CUSTOM_FIELDS_SITE_URI = "/v1/{site}/custom_fields";


	/**
	 * This method is used to create a new custom field.
	 *
	 * @param string $siteUrl					The site to which the custom field belongs to
	 * @param CustomFieldRequestData $data		The data sent to the API in order to create a custom field
	 * @param RequestOptions $requestOptions		The requestOptions to be used with the request
	 */
	public function create(string $siteUrl, CustomFieldRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::CUSTOM_FIELDS_SITE_URI);
		$urlBuilder->withPathSite($siteUrl);
		return $this->httpPost($urlBuilder, $requestOptions, $data);
	}


	/**
	 * Update an existing custom field by ID.
	 *
	 * @param string $id					The ID of the custom field to update
	 * @param CustomFieldRequestData $data	The data with which to update the specified custom field object
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function update(string $id, CustomFieldRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::CUSTOM_FIELDS_ID_URI);
		$urlBuilder->withPathId($id);
		return $this->httpPut($urlBuilder, $requestOptions, $data);
	}


	/**
	 * Clone a custom field
	 *
	 * @param string $id		The ID of the custom field to clone
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function cloneCustomField(string $id, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::CUSTOM_FIELDS_CLONE_URI);
		$urlBuilder->withPathId($id);
		return $this->httpPost($urlBuilder, $requestOptions);
	}


	/**
	 * Delete a custom field by it's ID
	 *
	 * @param string $id				The ID of the custom field to delete
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function delete(string $id, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::CUSTOM_FIELDS_ID_URI);
		$urlBuilder->withPathId($id);
		return $this->httpDelete($urlBuilder, $requestOptions);
	}


	/**
	 * Delete a list of custom fields by their ID
	 *
	 * @param DeleteRequestData $data		The list of IDs of the custom field to delete
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function deleteCustomFields(DeleteRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::CUSTOM_FIELDS_URI);
		return $this->httpDelete($urlBuilder, $requestOptions, $data);
	}


	/**
	 * Get a custom field by it's ID
	 *
	 * @param string $id			The ID of the custom field to retrieve
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function get(string $id, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::CUSTOM_FIELDS_ID_URI);
		$urlBuilder->withPathId($id);
		return $this->httpGet($urlBuilder, $requestOptions);
	}


	/**
	 * Get the list of custom fields for the specified site
	 *
	 * @param string $siteUrl	The site for which to retrieve the list of custom fields
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function getCustomFields(string $siteUrl, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::CUSTOM_FIELDS_SITE_URI);
		$urlBuilder->withPathSite($siteUrl);
		return $this->httpGet($urlBuilder, $requestOptions);
	}
}
