<?php
/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

declare(strict_types=1);

namespace Mambo\Coupon\Param;

use Mambo\Common\Param\AbstractServiceParams;

/**
 * Parameters for retrieving coupon states
 * @template-extends AbstractServiceParams<CouponGetStateListParams>
 */
class CouponGetStateListParams extends AbstractServiceParams
{

    /**
     * Specifies the page of results to retrieve.
     *
     * @param int $page
     * @return self
     */
    public function page(int $page): self
    {
        return $this->extraParams('page', (string) $page);
    }

    /**
     * Specifies the number of results to retrieve, up to a maximum of 100.
     *
     * @param int $count
     * @return self
     */
    public function count(int $count): self
    {
        return $this->extraParams('count', (string) $count);
    }

    /**
     * The list of tags to filter by (if any)
     *
     * @param array<string> $tags
     * @return self
     */
    public function tags(array $tags): self
    {
        return $this->extraParamsList('tags', $tags);
    }

    /**
     * Whether the tags should return items containing any one of the tags or
     * whether the tags should return only items containing all of the tags.
     * Possible values: hasAnyOf / hasAllOf Default: hasAnyOf
     *
     * @param string $tagsJoin
     * @return self
     */
    public function tagsJoin(string $tagsJoin): self
    {
        return $this->extraParams('tagsJoin', $tagsJoin);
    }

    /**
     * The tagUuid to use to filter the list by personalization tags
     *
     * @param string $tagUuid
     * @return self
     */
    public function tagUuid(string $tagUuid): self
    {
        return $this->extraParams('tagUuid', $tagUuid);
    }

    /**
     * Specifies the start date of the date range filter
     *
     * @param string $startDate
     * @return self
     */
    public function startDate(string $startDate): self
    {
        return $this->extraParams('startDate', $startDate);
    }

    /**
     * Specifies the end date of the date range filter
     *
     * @param string $endDate
     * @return self
     */
    public function endDate(string $endDate): self
    {
        return $this->extraParams('endDate', $endDate);
    }

    /**
     * Filter by awarded user IDs
     *
     * @param array<string> $awardedToUserIds
     * @return self
     */
    public function awardedToUserIds(array $awardedToUserIds): self
    {
        return $this->extraParamsList('awardedToUserIds', $awardedToUserIds);
    }

    /**
     * Filter by awarded user UUIDs
     *
     * @param array<string> $awardedToUuids
     * @return self
     */
    public function awardedToUuids(array $awardedToUuids): self
    {
        return $this->extraParamsList('awardedToUuids', $awardedToUuids);
    }

    /**
     * Filter by coupon IDs
     *
     * @param array<string> $couponIds
     * @return self
     */
    public function couponIds(array $couponIds): self
    {
        return $this->extraParamsList('couponIds', $couponIds);
    }

    /**
     * Filter by activity IDs
     *
     * @param array<string> $activityIds
     * @return self
     */
    public function activityIds(array $activityIds): self
    {
        return $this->extraParamsList('activityIds', $activityIds);
    }

    /**
     * Specifies whether only redeemable coupons should be returned
     *
     * @param bool $withRedeemableOnly
     * @return self
     */
    public function withRedeemableOnly(bool $withRedeemableOnly): self
    {
        return $this->extraParams('withRedeemableOnly', (string) $withRedeemableOnly);
    }

    /**
     * Specifies whether only redeemed coupons should be returned
     *
     * @param bool $withRedeemedOnly
     * @return self
     */
    public function withRedeemedOnly(bool $withRedeemedOnly): self
    {
        return $this->extraParams('withRedeemedOnly', (string) $withRedeemedOnly);
    }

    /**
     * Specifies whether only expired coupons should be returned
     *
     * @param bool $withExpiredOnly
     * @return self
     */
    public function withExpiredOnly(bool $withExpiredOnly): self
    {
        return $this->extraParams('withExpiredOnly', (string) $withExpiredOnly);
    }
}
