<?php
/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * This object captures the data required by the Coupon API in
 * order to create / update coupons.
 */

declare(strict_types=1);

namespace Mambo\Coupon\Data;

use Mambo\Common\Data\RequestData;
use Mambo\Common\Data\AbstractHasTagRequestData;
use Mambo\Common\Data\PointStore;
use Mambo\Common\Data\Scarcity;
use Mambo\Common\Data\Recurrence\Recurrence;

class CouponRequestData extends AbstractHasTagRequestData implements RequestData
{
	/**
	 * Whether the image associated to the Coupon should be removed
	 * @return boolean removeImage
	 */
	public function getRemoveImage()
	{
		return $this->data['removeImage'];
	}
	public function setRemoveImage($removeImage)
	{
		$this->data['removeImage'] = $removeImage;
	}

	/**
	 * The name of the coupon.
	 */
	public function getName()
	{
		return $this->data['name'];
	}
	public function setName($name)
	{
		$this->data['name'] = $name;
	}

	/**
	 * This field contains information about the couponCode associated
	 * with this coupon.
	 */
	public function getCouponCode()
	{
		return $this->data['couponCode'];
	}
	public function setCouponCode(CouponCode $couponCode)
	{
		$this->data['couponCode'] = $couponCode;
	}

	/**
	 * The type of coupon. Valid values include: percent, fixed, shipping and custom.
	 */
	public function getType()
	{
		return $this->data['type'];
	}
	public function setType($type)
	{
		$this->data['type'] = $type;
	}

	/**
	 * This flag indicates whether this coupon is active.
	 */
	public function getActive()
	{
		return $this->data['active'];
	}
	public function setActive($active)
	{
		$this->data['active'] = $active;
	}

	/**
	 * In the case of a percent or fixed based coupon, the amount should contain
	 * the percentage or currency amount that the coupon provides.
	 */
	public function getAmount()
	{
		return $this->data['amount'];
	}
	public function setAmount($amount)
	{
		$this->data['amount'] = $amount;
	}

	/**
	 * In the case of a custom based coupon, this field should contain information
	 * relating to the nature of the coupon that is being given. For example: buy 1
	 * get 1 free.
	 */
	public function getCustomPrize()
	{
		return $this->data['custom'];
	}
	public function setCustomPrize($custom)
	{
		$this->data['custom'] = $custom;
	}

	/**
	 * The scarcity represents the overall number of times that a particular
	 * coupon can be unlocked. This effectively creates a limited series
	 * coupon which can only be unlocked X times.
	 * See the coupon page in administration panel for more information.
	 */
	public function getScarcity()
	{
		return $this->data['scarcity'];
	}
	public function setScarcity(Scarcity $scarcity)
	{
		$this->data['scarcity'] = $scarcity;
	}

	/**
	 * The buy criteria defines whether the coupon can be bought with
	 * points and how the buying process works.
	 * If the coupon cannot be bought then this value should be
	 * set to null.
	 */
	public function getBuyCriteria()
	{
		return $this->data['buyCriteria'];
	}
	public function setBuyCriteria(BuyCriteria $buyCriteria)
	{
		$this->data['buyCriteria'] = $buyCriteria;
	}

	/**
	 * The message to be displayed in the coupon purchasing screen. This can provide
	 * more information about the coupon, how to redeem it, any exceptions for the
	 * coupon, etc.
	 */
	public function getHelpMessage()
	{
		return $this->data['helpMessage'];
	}
	public function setHelpMessage($helpMessage)
	{
		$this->data['helpMessage'] = $helpMessage;
	}

	/**
	 * The usage indicates how many times this coupon can be used. Valid values
	 * include: single or multi. If single is used then the coupon can only be used
	 * once. If multi is used then the coupon can be re-used multiple times (if an
	 * expire period is set, then it can be used re-used until it's expired).
	 *
	 * Important: users can be awarded with the same single use coupon more than
	 * once. This will award the same coupon twice to the user and they should be
	 * able to redeem BOTH those rewards.
	 *
	 */
	public function getUsage()
	{
		return $this->data['usage'];
	}
	public function setUsage($usage)
	{
		$this->data['usage'] = $usage;
	}

	/**
	 * Defines whether this coupon expires or not. If it does, then the relevant time
	 * frame needs to be set.
	 * See the {@link NeverExpiration}, {@link FixedDateExpiration} or
	 * {@link VariablePeriodExpiration} objects for more information.
	 */
	public function getExpiration()
	{
		return $this->data['expiration'];
	}
	public function setExpiration(Recurrence $expiration)
	{
		$this->data['expiration'] = $expiration;
	}
	
	
	/**
	 * Custom fields defined for the coupon. These can contain additional
	 * data or any kind of information you would like to store which isn't a
	 * standard field of the coupon.
	 */
	public function getCustomFields()
	{
		return $this->data['customFields'];
	}
	public function setCustomFields(array $customFields)
	{
		$this->data['customFields'] = $customFields;
	}


	/**
	 * This contains the list of the translations which must be used with the object.
	 * Setting a property directly on the object (such as using setName()) will add
	 * the property to the list of translations using the default language of the server.
	 * If you set a property on the object directly (such as setName()) and then set
	 * the same property using setTranslations(), then the setTranslations() property
	 * will take priority.
	 */
	public function getTranslations()
	{
		return $this->data['translations'];
	}
	public function setTranslations(array $translations)
	{
		$this->data['translations'] = $translations;
	}
	public function addTranslation(CouponTranslation $translation)
	{
		if (!isset($this->data['translations'])) {
			$this->data['translations'] = array();
		}
		array_push($this->data['translations'], $translation);
	}


	/**
	 * Return the JSON string equivalent of this object
	 */
	public function getJsonString(): string
	{
		$json = $this->data;

		if (isset($json['couponCode'])) {
			$json['couponCode'] = $json['couponCode']->getJsonArray();
		}

		if (isset($json['buyCriteria'])) {
			$json['buyCriteria'] = $json['buyCriteria']->getJsonArray();
		}

		if (isset($json['expiration'])) {
			$json['expiration'] = $json['expiration']->getJsonArray();
		}

		if (isset($json['scarcity'])) {
			$json['scarcity'] = $json['scarcity']->getJsonArray();
		}
		
		if (isset($json['customFields'])) {
			$customFieldsArr = array();
			foreach ($json['customFields'] as $customField) {
				array_push($customFieldsArr, $customField->getJsonArray());
			}
			$json['customFields'] = $customFieldsArr;
		}

		if (isset($json['translations'])) {
			$translationsArr = array();
			foreach ($json['translations'] as $translation) {
				array_push($translationsArr, $translation->getJsonArray());
			}
			$json['translations'] = $translationsArr;
		}

		return json_encode($json);
	}
}
