<?php
/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * The MamboCouponsService class handles all Coupon related requests
 * to the Mambo API.
 */

declare(strict_types=1);

namespace Mambo\Coupon;

use Mambo\Coupon\Data\CouponRequestData;
use Mambo\Coupon\Data\CouponUserRequestData;
use Mambo\Coupon\Param\CouponGetListParams;
use Mambo\Coupon\Param\CouponGetRegularListParams;
use Mambo\Coupon\Param\CouponGetBuyableListParams;
use Mambo\Coupon\Param\CouponGetStateListParams;
use Mambo\Common\Service\AbstractService;
use Mambo\Common\Data\DeleteRequestData;
use Mambo\Http\ApiUrlBuilder;
use Mambo\Http\RequestOptions;
use Mambo\Util\Preconditions;

class CouponsService extends AbstractService
{
	/**
	 * Coupon service end point URI
	 * @var string
	 */
	const COUPONS_URI = "/v1/coupons";
	const COUPONS_ID_URI = "/v1/coupons/{id}";
	const COUPONS_IMAGE_URI = "/v1/coupons/{id}/image";
	const COUPONS_CLONE_URI = "/v1/coupons/{id}/clone";
	const COUPONS_CUSTOM_URI = "/v1/coupons/{id}/custom_fields";
	const COUPONS_CUSTOM_IMAGE_URI = "/v1/coupons/{id}/custom_fields/image";

	const COUPONS_SITE_URI = "/v1/{site}/coupons";
	const COUPONS_BUYABLE_URI = "/v1/{site}/coupons/buyable";
	const COUPONS_REGULAR_URI = "/v1/{site}/coupons/regular";
	const COUPONS_STATE_URI = "/v1/{site}/coupons/states";
	const VALIDATE_COUPON_URI = "/v1/{site}/coupons/validate";


	/**
	 * This method is used to create a new coupon.
	 *
	 * @param string $siteUrl					The site to which the coupon belongs to
	 * @param CouponRequestData $data			The data sent to the API in order to create a coupon
	 * @param RequestOptions $requestOptions		The requestOptions to be used with the request
	 */
	public function create(string $siteUrl, CouponRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::COUPONS_SITE_URI);
		$urlBuilder->withPathSite($siteUrl);
		return $this->httpPost($urlBuilder, $requestOptions, $data);
	}


	/**
	 * Update an existing coupon by ID.
	 *
	 * @param string $id					The ID of the coupon to update
	 * @param CouponRequestData $data	The data with which to update the specified coupon object
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function update(string $id, CouponRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::COUPONS_ID_URI);
		$urlBuilder->withPathId($id);
		return $this->httpPut($urlBuilder, $requestOptions, $data);
	}


	/**
	 * Upload an image for the coupon
	 *
	 * @param string $id		The coupon for which to upload the image
	 * @param string|false $image 	The image to upload for the coupon
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function uploadImage(string $id, string|false $image, RequestOptions $requestOptions = null): array|object
	{
		Preconditions::checkNotEmpty($image, "The image must not be empty");

		$urlBuilder = new ApiUrlBuilder(self::COUPONS_IMAGE_URI);
		$urlBuilder->withPathId($id);

		return $this->httpUpload($urlBuilder, $image, $requestOptions);
	}


	/**
	 * Clone a coupon
	 *
	 * @param string $id		The ID of the coupon to clone
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function cloneCoupon(string $id, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::COUPONS_CLONE_URI);
		$urlBuilder->withPathId($id);
		return $this->httpPost($urlBuilder, $requestOptions);
	}


	/**
	 * Delete a coupon by it's ID
	 *
	 * @param string $id				The ID of the coupon to delete
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function delete(string $id, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::COUPONS_ID_URI);
		$urlBuilder->withPathId($id);
		return $this->httpDelete($urlBuilder, $requestOptions);
	}


	/**
	 * Delete a list of coupons by their ID
	 *
	 * @param DeleteRequestData $data		The list of IDs of the coupon to delete
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function deleteCoupons(DeleteRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::COUPONS_URI);
		return $this->httpDelete($urlBuilder, $requestOptions, $data);
	}


	/**
	 * Get a coupon by its ID
	 *
	 * @param string $id			The ID of the coupon to retrieve
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function get(string $id, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::COUPONS_ID_URI);
		$urlBuilder->withPathId($id);
		return $this->httpGet($urlBuilder, $requestOptions);
	}


	/**
	 * This method is used to validate a coupon code can be redeemed for the specified user.
	 *
	 * @param string $siteUrl		The site to which the user and coupon belong to
	 * @param string $uuid			The Unique User ID of the user against which we will validate the coupon code
	 * @param string $couponCode		The code of the coupon we should validate against the specified user
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function validateRedeem(string $siteUrl, string $uuid, string $couponCode, RequestOptions $requestOptions = null): array|object
	{
		$data = new CouponUserRequestData();
		$data->setUuid($uuid);
		$data->setCouponCode($couponCode);

		$urlBuilder = new ApiUrlBuilder(self::VALIDATE_COUPON_URI);
		$urlBuilder->withPathSite($siteUrl);

		return $this->httpPost($urlBuilder, $requestOptions, $data);
	}


	/**
	 * Get the list of coupons for the specified site
	 *
	 * @param string $siteUrl					The site for which to retrieve the list of coupons
	 * @param CouponGetListParams $params		The query parameters to be used with the request
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function getCoupons(string $siteUrl, CouponGetListParams $params = null, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::COUPONS_SITE_URI);
		$urlBuilder->withPathSite($siteUrl);
		$urlBuilder->withQueryParams($params);

		return $this->httpGet($urlBuilder, $requestOptions);
	}


	/**
	 * Get the list of buyable coupons for the specified site
	 *
	 * @param string $siteUrl						The site for which to retrieve the list of buyable coupons
	 * @param CouponGetBuyableListParams $params	The query parameters to be used with the request
	 * @param RequestOptions $requestOptions		The requestOptions to be used with the request
	 */
	public function getBuyableCoupons(string $siteUrl, CouponGetBuyableListParams $params = null, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::COUPONS_BUYABLE_URI);
		$urlBuilder->withPathSite($siteUrl);
		$urlBuilder->withQueryParams($params);

		return $this->httpGet($urlBuilder, $requestOptions);
	}


	/**
	 * Get the list of regular (non-buyable) coupons for the specified site
	 *
	 * @param string $siteUrl						The site for which to retrieve the list of regular (non-buyable) coupons
	 * @param CouponGetRegularListParams $params	The query parameters to be used with the request
	 * @param RequestOptions $requestOptions		The requestOptions to be used with the request
	 */
	public function getRegularCoupons(string $siteUrl, CouponGetRegularListParams $params = null, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::COUPONS_REGULAR_URI);
		$urlBuilder->withPathSite($siteUrl);
		$urlBuilder->withQueryParams($params);

		return $this->httpGet($urlBuilder, $requestOptions);
	}


	/**
	 * Get the list of coupons states for the specified site
	 *
	 * @param string $siteUrl					The site for which to retrieve the list of awarded coupons
	 * @param CouponGetStateListParams $params	The query parameters to be used with the request
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request

	 */
	public function getCouponStates(
		string $siteUrl,
		CouponGetStateListParams $params = null,
		RequestOptions $requestOptions = null
	): array|object {

		$urlBuilder = new ApiUrlBuilder(self::COUPONS_STATE_URI);
		$urlBuilder->withPathSite($siteUrl);
		$urlBuilder->withQueryParams($params);

		return $this->httpGet($urlBuilder, $requestOptions);
	}
	
	
	/**
	 * This method is used to add custom fields to an existing coupon
	 *
	 * @param string $id						The ID of the coupon to update
	 * @param CustomFieldValueRequestData $data	The list of custom fields to add
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function addCustomFields(string $id, CustomFieldValueRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		return $this->modCustomFields(HttpClient::POST, $id, $data, $requestOptions);
	}

	
	/**
	 * This method is used to update custom fields in an existing coupon
	 *
	 * @param string $id						The ID of the coupon to update
	 * @param CustomFieldValueRequestData $data	The list of custom fields to update
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function updateCustomFields(string $id, CustomFieldValueRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		return $this->modCustomFields(HttpClient::PUT, $id, $data, $requestOptions);
	}

	
	/**
	 * This method is used to delete custom fields in an existing coupon
	 *
	 * @param string $id						The ID of the coupon to update
	 * @param CustomFieldValueRequestData $data	The list of custom fields to update
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function deleteCustomFields(string $id, CustomFieldValueRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		return $this->modCustomFields(HttpClient::DELETE, $id, $data, $requestOptions);
	}

	
	/**
	 * Modifying custom fields
	 */
	private function modCustomFields(string $method, string $id, CustomFieldValueRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::COUPONS_CUSTOM_URI);
		$urlBuilder->withPathId($id);
		return $this->client->request($urlBuilder, $method, $data, $requestOptions);
	}

	
	/**
	 * This method is used to add a custom field with an image in an existing coupon
	 *
	 * @param string $id						The ID of the coupon to update
	 * @param CustomFieldValue $data         The custom field to update
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function addCustomField(string $id, CustomFieldValue $data, string|false $image, RequestOptions $requestOptions = null): array|object
	{
		Preconditions::checkNotEmpty($image, "The image must not be empty");

		$urlBuilder = new ApiUrlBuilder(self::COUPONS_CUSTOM_IMAGE_URI);
		$urlBuilder->withPathId($id);

		return $this->httpUpload($urlBuilder, $image, $requestOptions, $data);
	}
}
