<?php
/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * This class is extended by all the Mambo API service classes and
 * provides basic functionality to all services.
 */

declare(strict_types=1);

namespace Mambo\Common\Service;

use Mambo\Common\Data\RequestData;
use Mambo\Http\ApiUrlBuilder;
use Mambo\Http\RequestOptions;
use Mambo\Http\HttpClient;

abstract class AbstractService
{

    /**
     * The client used to connect to the API
     * @var HttpClient
     */
    protected HttpClient $client;


    public function __construct(HttpClient $client)
    {
        $this->client = $client;
    }


    /**
     * @param ApiUrlBuilder $urlBuilder         The URL builder that generates the relevant URL
     * @param string|false $image               The image to be sent with the request
     * @param RequestOptions $requestOptions    The requestOptions to be used with the request
     * @param RequestData $data                 The data to be sent with the request
     */
    protected function httpUpload(ApiUrlBuilder $urlBuilder, string|false $image, ?RequestOptions $requestOptions, ?RequestData $data = null): array|object
    {
        return $this->client->upload($urlBuilder, $image, $data, $requestOptions);
    }


    /**
     * @param ApiUrlBuilder $urlBuilder         The URL builder that generates the relevant URL
     * @param RequestOptions $requestOptions    The requestOptions to be used with the request
     * @param RequestData $data                 The data to be sent with the request
     */
    protected function httpPost(ApiUrlBuilder $urlBuilder, ?RequestOptions $requestOptions, ?RequestData $data = null): array|object
    {
        return $this->client->request($urlBuilder, HttpClient::POST, $data, $requestOptions);
    }


    /**
     * @param ApiUrlBuilder $urlBuilder         The URL builder that generates the relevant URL
     * @param RequestOptions $requestOptions    The requestOptions to be used with the request
     * @param RequestData $data                 The data to be sent with the request
     */
    protected function httpPut(ApiUrlBuilder $urlBuilder, ?RequestOptions $requestOptions, ?RequestData $data = null): array|object
    {
        return $this->client->request($urlBuilder, HttpClient::PUT, $data, $requestOptions);
    }


    /**
     * @param ApiUrlBuilder $urlBuilder         The URL builder that generates the relevant URL
     * @param RequestOptions $requestOptions    The requestOptions to be used with the request
     * @param RequestData $data                 The data to be sent with the request
     */
    protected function httpDelete(ApiUrlBuilder $urlBuilder, ?RequestOptions $requestOptions, ?RequestData $data = null): array|object
    {
        return $this->client->request($urlBuilder, HttpClient::DELETE, $data, $requestOptions);
    }


    /**
     * @param ApiUrlBuilder $urlBuilder         The URL builder that generates the relevant URL
     * @param RequestOptions $requestOptions    The requestOptions to be used with the request
     */
    protected function httpGet(ApiUrlBuilder $urlBuilder, ?RequestOptions $requestOptions): array|object
    {
        return $this->client->request($urlBuilder, HttpClient::GET, null, $requestOptions);
    }
}
