<?php
/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

declare(strict_types=1);

namespace Mambo\Common\Param;

use Mambo\Util\Preconditions;

/**
 * Abstract service params with shared methods and properties
 * @template P
 */
abstract class AbstractServiceParams implements ServiceParams
{
    /** @var array<string, array<string>> */
    protected $params = [];

    /**
     * Constructor to initialize the parameters.
     *
     * @param array<string, mixed> $inputParams The initial input parameters
     * @throws \InvalidArgumentException
     */
    public function __construct(array $inputParams = [])
    {
        if (!empty($inputParams)) {
            $this->convertAndStoreParams($inputParams);
        }
    }

    /**
     * Converts the given array to the appropriate structure for storing in params.
     *
     * @param array<string, mixed> $input The original input array
     * @return void
     * @throws \InvalidArgumentException
     */
    private function convertAndStoreParams(array $input): void
    {
        foreach ($input as $key => $value) {
            if (is_array($value)) {
            	$value = array_map([$this, 'convertToString'], $value);
            } else {
            	$value = [$this->convertToString($value)];
            }

            $this->extraParamsList($key, $value);
        }
    }

	/**
	* @param mixed $value
	* @return string
	*/
	private function convertToString($value): string 
	{
		if (is_bool($value)) {
			return $value ? 'true' : 'false'; 
		}
		return strval($value);
	}

    /**
     * @param string $key
     * @param mixed $value
     * @return P
     * @throws \InvalidArgumentException
     */
    public function extraParams(string $key, mixed $value)
    {
        Preconditions::checkNotNull($key, "key must not be null");
        Preconditions::checkNotNull($value, "value must not be null");

    	$this->params[$key] = [$this->convertToString($value)];
        return $this;
    }

    /**
     * @param string $key
     * @param array<mixed> $values
     * @return P
     * @throws \InvalidArgumentException
     */
    public function extraParamsList(string $key, array $values)
    {
        Preconditions::checkNotNull($key, "key must not be null");
        Preconditions::checkNotNull($values, "values must not be null");

   		$this->params[$key] = array_map([$this, 'convertToString'], $values);
        return $this;
    }

    /**
     * @return array<string, array<string>>
     */
    public function asMap(): array
    {
        return $this->params;
    }
}
