<?php
/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * Represents an object which recurs on a fixed weekly interval.
 * For example:
 * - Weekly on Tuesday at 11:00AM
 */

declare(strict_types=1);

namespace Mambo\Common\Data\Recurrence\Period;

class FixedPeriodWeekly implements FixedPeriod
{
	private $data = array();


	/**
	 * The type of fixed period: weekly
	 * This field cannot be null.
	 */
	public function getType()
	{
		return 'weekly';
	}


	/**
	 * The hour indicates the hour on which the object should recur.
	 * Valid values range from 0 to 23 where 0 indicates midnight.
	 * This field cannot be null.
	 */
	public function getHour()
	{
		return $this->data['hour'];
	}
	public function setHour($hour)
	{
		$this->data['hour'] = $hour;
	}


	/**
	 * The days indicates the days of the week on which the object should recur.
	 * Valid values range from 0 to 6 where 0 indicates Monday and 6 indicates Sunday.
	 * This field cannot be null.
	 */
	public function getDaysOfWeek()
	{
		return $this->data['daysOfWeek'];
	}
	public function setDaysOfWeek(array $daysOfWeek)
	{
		$this->data['daysOfWeek'] = $daysOfWeek;
	}


	/**
	 * The week interval indicates every how many weeks the object should recur.
	 * Valid values range from 1 to 52.
	 * This field cannot be null.
	 */
	public function getWeekInterval()
	{
		return $this->data['weekInterval'];
	}
	public function setWeekInterval($weekInterval)
	{
		$this->data['weekInterval'] = $weekInterval;
	}


	/**
	 * The period start date indicates the date from which the interval should start.
	 * Note that only the date portion of the time stamp will be used.
	 * This will be a UTC time stamp in ISO 8601 format with
	 * millisecond precision: YYYY-MM-DDTHH:MM:SS.MMMZ<br>
	 * For example: 2013-01-20T20:43:24.094Z
	 * This field cannot be null.
	 */
	public function getPeriodStart()
	{
		return $this->data['periodStart'];
	}
	public function setPeriodStart($periodStart)
	{
		$this->data['periodStart'] = $periodStart;
	}


	/**
	 * Returns the current model as an array ready to
	 * be json_encoded
	 */
	public function getJsonArray(): array
	{
		$json = $this->data;
		$json['type'] = $this->getType();
		return $json;
	}
}
