<?php
/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * Properties used to configuration the HttpClient
 */

/**
 * Properties used to configure the HttpClient
 */

declare(strict_types=1);

namespace Mambo;

use Mambo\Http\MamboCredentials;

class ClientConfiguration
{
    /** @var string */
    private $serverBaseUrl = "https://api.mambo.io";

    /** @var bool */
    private $debug = false;

    /** @var string */
    private $acceptLanguage = "en";

    /** @var bool */
    private $jsonDecodeAsArray = false;

    /** @var int */
    private $timeoutMilliseconds = 60000;

    /** @var int */
    private $connectTimeoutMilliseconds = 20000;

    /** @var string|null */
    private $publicKey;

    /** @var string|null */
    private $privateKey;

    /**
     * Get the base URL of the server
     * @return string
     */
    public function getServerBaseUrl(): string
    {
        return $this->serverBaseUrl;
    }

    /**
     * Set the base URL of the server
     * @param string $baseUrl
     * @return self
     * @throws \InvalidArgumentException
     */
    public function setServerBaseUrl(string $baseUrl): self
    {
        if (strpos($baseUrl, "http") !== 0) {
            throw new \InvalidArgumentException("Please prefix the URL with the relevant protocol (e.g. https://)");
        }

        $this->serverBaseUrl = rtrim($baseUrl, "/");
        return $this;
    }

    /**
     * Get whether debugging is enabled
     * @return bool
     */
    public function getDebug(): bool
    {
        return $this->debug;
    }

    /**
     * Set whether debugging is enabled
     * @param bool $debug
     * @return self
     */
    public function setDebug(bool $debug): self
    {
        $this->debug = $debug;
        return $this;
    }

    /**
     * Get the accept language header value
     * @return string
     */
    public function getAcceptLanguage(): string
    {
        return $this->acceptLanguage;
    }

    /**
     * Set the accept language header value. Currently supported values are: en / pt
     * @param string $acceptLanguage
     * @return self
     */
    public function setAcceptLanguage(string $acceptLanguage): self
    {
        $this->acceptLanguage = $acceptLanguage;
        return $this;
    }

    /**
     * Get whether JSON should be decoded as an array
     * @return bool
     */
    public function getJsonDecodeAsArray(): bool
    {
        return $this->jsonDecodeAsArray;
    }

    /**
     * Set whether JSON should be decoded as an array
     * @param bool $jsonDecodeAsArray
     * @return self
     */
    public function setJsonDecodeAsArray(bool $jsonDecodeAsArray): self
    {
        $this->jsonDecodeAsArray = $jsonDecodeAsArray;
        return $this;
    }

    /**
     * Get the timeout in seconds to read a response from the server
     * @return int
     */
    public function getTimeoutMilliseconds(): int
    {
        return $this->timeoutMilliseconds;
    }

    /**
     * Set the timeout in seconds to read a response from the server
     * @param int $timeout
     * @return self
     */
    public function setTimeoutMilliseconds(int $timeout): self
    {
        $this->timeoutMilliseconds = $timeout;
        return $this;
    }

    /**
     * Get the timeout in seconds to connect to the server
     * @return int
     */
    public function getConnectTimeoutMilliseconds(): int
    {
        return $this->connectTimeoutMilliseconds;
    }

    /**
     * Set the timeout in seconds to connect to the server
     * @param int $timeout
     * @return self
     */
    public function setConnectTimeoutMilliseconds(int $timeout): self
    {
        $this->connectTimeoutMilliseconds = $timeout;
        return $this;
    }

    /**
     * Get the credentials
     * @return MamboCredentials
     */
    public function getCredentials(): MamboCredentials
    {
        return new MamboCredentials( $this->publicKey, $this->privateKey );
    }

    /**
     * Set the API credentials
     * @param string $publicKey
     * @param string $privateKey
     * @return self
     */
    public function setCredentials(string $publicKey, string $privateKey): self
    {
        $this->publicKey = $publicKey;
        $this->privateKey = $privateKey;
        return $this;
    }
}
