<?php
/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * The MamboBehavioursService class handles all Behaviour related requests
 * to the Mambo API.
 */

declare(strict_types=1);

namespace Mambo\Behaviour;

use Mambo\Behaviour\Data\BehaviourRequestData;
use Mambo\Behaviour\Param\BehaviourGetListParams;
use Mambo\Behaviour\Param\BehaviourGetSimpleListParams;
use Mambo\Behaviour\Param\BehaviourGetFlexibleListParams;
use Mambo\Common\Service\AbstractService;
use Mambo\Common\Data\DeleteRequestData;
use Mambo\Common\Data\CustomFieldValueRequestData;
use Mambo\Common\Data\CustomFieldValue;
use Mambo\Http\ApiUrlBuilder;
use Mambo\Http\RequestOptions;
use Mambo\Http\HttpClient;
use Mambo\Util\Preconditions;

class BehavioursService extends AbstractService
{
	/**
	 * Behaviour service end point URI
	 * @var string
	 */
	const BEHAVIOURS_URI = "/v1/behaviours";
	const BEHAVIOURS_ID_URI = "/v1/behaviours/{id}";
	const BEHAVIOURS_IMAGE_URI = "/v1/behaviours/{id}/image";
	const BEHAVIOURS_CLONE_URI = "/v1/behaviours/{id}/clone";
	const BEHAVIOURS_CUSTOM_URI = "/v1/behaviours/{id}/custom_fields";
	const BEHAVIOURS_CUSTOM_IMAGE_URI = "/v1/behaviours/{id}/custom_fields/image";

	const BEHAVIOURS_SITE_URI = "/v1/{site}/behaviours";
	const SIMPLE_BEHAVIOURS_SITE_URI = "/v1/{site}/behaviours/simple";
	const FLEXIBLE_BEHAVIOURS_SITE_URI = "/v1/{site}/behaviours/flexible";


	/**
	 * This method is used to create a new behaviour.
	 *
	 * @param string $siteUrl					The site to which the behaviour belongs to
	 * @param BehaviourRequestData $data			The data sent to the API in order to create a behaviour
	 * @param RequestOptions $requestOptions		The requestOptions to be used with the request
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function create(string $siteUrl, BehaviourRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::BEHAVIOURS_SITE_URI);
		$urlBuilder->withPathSite($siteUrl);
		return $this->httpPost($urlBuilder, $requestOptions, $data);
	}


	/**
	 * Update an existing behaviour by ID.
	 *
	 * @param string $id					The ID of the behaviour to update
	 * @param BehaviourRequestData $data	The data with which to update the specified behaviour object
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function update(string $id, BehaviourRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::BEHAVIOURS_ID_URI);
		$urlBuilder->withPathId($id);
		return $this->httpPut($urlBuilder, $requestOptions, $data);
	}


	/**
	 * Upload an image for the behaviour
	 *
	 * @param string $id		The behaviour for which to upload the image
	 * @param string|false $image 	The image to upload for the behaviour
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function uploadImage(string $id, string|false $image, RequestOptions $requestOptions = null): array|object
	{
		Preconditions::checkNotEmpty($image, "image must not be empty");

		$urlBuilder = new ApiUrlBuilder(self::BEHAVIOURS_IMAGE_URI);
		$urlBuilder->withPathId($id);
		return $this->httpUpload($urlBuilder, $image, $requestOptions);
	}


	/**
	 * Clone a behaviour
	 *
	 * @param string $id		The ID of the behaviour to clone
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function cloneBehaviour(string $id, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::BEHAVIOURS_CLONE_URI);
		$urlBuilder->withPathId($id);
		return $this->httpPost($urlBuilder, $requestOptions);
	}


	/**
	 * Delete a behaviour by it's ID
	 *
	 * @param string $id		The ID of the behaviour to delete
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function delete(string $id, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::BEHAVIOURS_ID_URI);
		$urlBuilder->withPathId($id);
		return $this->httpDelete($urlBuilder, $requestOptions);
	}


	/**
	 * Delete a list of behaviours by their ID
	 *
	 * @param DeleteRequestData $data		The list of IDs of the behaviour to delete
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function deleteBehaviours(DeleteRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::BEHAVIOURS_URI);
		return $this->httpDelete($urlBuilder, $requestOptions, $data);
	}


	/**
	 * Get a behaviour by it's ID
	 *
	 * @param string $id			The ID of the behaviour to retrieve
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function get(string $id, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::BEHAVIOURS_ID_URI);
		$urlBuilder->withPathId($id);
		return $this->httpGet($urlBuilder, $requestOptions);
	}


	/**
	 * Get the list of behaviours for the specified site
	 *
	 * @param string $siteUrl					The site for which to retrieve the list of behaviours
	 * @param BehaviourGetListParams $params	The query parameters to be used with the request
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function getBehaviours(
		string $siteUrl,
		BehaviourGetListParams $params = null,
		RequestOptions $requestOptions = null
	): array|object {

		$urlBuilder = new ApiUrlBuilder(self::BEHAVIOURS_SITE_URI);
		$urlBuilder->withPathSite($siteUrl);
		$urlBuilder->withQueryParams($params);

		return $this->httpGet($urlBuilder, $requestOptions);
	}


	/**
	 * Get the list of simple behaviours for the specified site
	 *
	 * @param string $siteUrl						The site for which to retrieve the list of simple behaviours
	 * @param BehaviourGetSimpleListParams $params	The query parameters to be used with the request
	 * @param RequestOptions $requestOptions		The requestOptions to be used with the request
	 */
	public function getSimpleBehaviours(
		string $siteUrl,
		BehaviourGetSimpleListParams $params = null,
		RequestOptions $requestOptions = null
	): array|object {

		$urlBuilder = new ApiUrlBuilder(self::SIMPLE_BEHAVIOURS_SITE_URI);
		$urlBuilder->withPathSite($siteUrl);
		$urlBuilder->withQueryParams($params);

		return $this->httpGet($urlBuilder, $requestOptions);
	}


	/**
	 * Get the list of flexible behaviours for the specified site
	 *
	 * @param string $siteUrl							The site for which to retrieve the list of flexible behaviours
	 * @param BehaviourGetFlexibleListParams $params	The query parameters to be used with the request
	 * @param RequestOptions $requestOptions			The requestOptions to be used with the request
	 */
	public function getFlexibleBehaviours(
		string $siteUrl,
		BehaviourGetFlexibleListParams $params = null,
		RequestOptions $requestOptions = null
	): array|object {

		$urlBuilder = new ApiUrlBuilder(self::FLEXIBLE_BEHAVIOURS_SITE_URI);
		$urlBuilder->withPathSite($siteUrl);
		$urlBuilder->withQueryParams($params);

		return $this->httpGet($urlBuilder, $requestOptions);
	}


	/**
	 * This method is used to add custom fields to an existing behaviour
	 *
	 * @param string $id						The ID of the behaviour to update
	 * @param CustomFieldValueRequestData $data	The list of custom fields to add
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function addCustomFields(string $id, CustomFieldValueRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		return $this->modCustomFields(HttpClient::POST, $id, $data, $requestOptions);
	}


	/**
	 * This method is used to update custom fields in an existing behaviour
	 *
	 * @param string $id						The ID of the behaviour to update
	 * @param CustomFieldValueRequestData $data	The list of custom fields to update
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function updateCustomFields(string $id, CustomFieldValueRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		return $this->modCustomFields(HttpClient::PUT, $id, $data, $requestOptions);
	}


	/**
	 * This method is used to delete custom fields in an existing behaviour
	 *
	 * @param string $id						The ID of the behaviour to update
	 * @param CustomFieldValueRequestData $data	The list of custom fields to update
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function deleteCustomFields(string $id, CustomFieldValueRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		return $this->modCustomFields(HttpClient::DELETE, $id, $data, $requestOptions);
	}


	/**
	 * Modifying custom fields
	 */
	private function modCustomFields(string $method, string $id, CustomFieldValueRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::BEHAVIOURS_CUSTOM_URI);
		$urlBuilder->withPathId($id);
		return $this->client->request($urlBuilder, $method, $data, $requestOptions);
	}


	/**
	 * This method is used to add a custom field with an image in an existing behaviour
	 *
	 * @param string $id						The ID of the behaviour to update
	 * @param CustomFieldValue $data	        The custom field to update
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function addCustomField(string $id, CustomFieldValue $data, string|false $image, RequestOptions $requestOptions = null): array|object
	{
		Preconditions::checkNotEmpty($image, "image must not be empty");

		$urlBuilder = new ApiUrlBuilder(self::BEHAVIOURS_CUSTOM_IMAGE_URI);
		$urlBuilder->withPathId($id);
		return $this->httpUpload($urlBuilder, $image, $requestOptions, $data);
	}
}
