<?php
/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

declare(strict_types=1);

namespace Mambo\Activity\Param;

use Mambo\Common\Param\AbstractServiceParams;
use InvalidArgumentException;

/**
 * Parameters for retrieving a list of activities
 * @extends AbstractServiceParams<ActivityGetListParams>
 */
class ActivityGetListParams extends AbstractServiceParams
{

    /**
     * Specifies the page of results to retrieve.
     *
     * @param int $page
     * @return self
     */
    public function page(int $page): self
    {
        return $this->extraParams('page', $page);
    }

    /**
     * Specifies the number of results to retrieve, up to a maximum of 100.
     *
     * @param int $count
     * @return self
     */
    public function count(int $count): self
    {
        return $this->extraParams('count', $count);
    }

    /**
     * The list of tags to filter by (if any)
     *
     * @param array<string> $tags
     * @return self
     */
    public function tags(array $tags): self
    {
        return $this->extraParamsList('tags', $tags);
    }

    /**
     * Whether the tags should return items containing any one of the tags or all tags
     *
     * @param string $tagsJoin Possible values: hasAnyOf / hasAllOf Default: hasAnyOf
     * @return self
     */
    public function tagsJoin(string $tagsJoin): self
    {
        return $this->extraParams('tagsJoin', $tagsJoin);
    }

    /**
     * The tagUuid to use to filter the list by personalization tags
     *
     * @param string $tagUuid
     * @return self
     */
    public function tagUuid(string $tagUuid): self
    {
        return $this->extraParams('tagUuid', $tagUuid);
    }

    /**
     * Specifies whether only activities which unlocked behaviors should be returned
     *
     * @param bool $withBehavioursOnly
     * @return self
     */
    public function withBehavioursOnly(bool $withBehavioursOnly): self
    {
        return $this->extraParams('withBehavioursOnly', $withBehavioursOnly);
    }

    /**
     * Specifies whether only activities which unlocked rewards should be returned
     *
     * @param bool $withRewardsOnly
     * @return self
     */
    public function withRewardsOnly(bool $withRewardsOnly): self
    {
        return $this->extraParams('withRewardsOnly', $withRewardsOnly);
    }

    /**
     * Specifies whether only activities which unlocked missions should be returned
     *
     * @param bool $withMissionsOnly
     * @return self
     */
    public function withMissionsOnly(bool $withMissionsOnly): self
    {
        return $this->extraParams('withMissionsOnly', $withMissionsOnly);
    }

    /**
     * Specifies whether only activities which unlocked levels should be returned
     *
     * @param bool $withLevelsOnly
     * @return self
     */
    public function withLevelsOnly(bool $withLevelsOnly): self
    {
        return $this->extraParams('withLevelsOnly', $withLevelsOnly);
    }

    /**
     * Specifies whether only activities which unlocked achievements should be returned
     *
     * @param bool $withAchievementsOnly
     * @return self
     */
    public function withAchievementsOnly(bool $withAchievementsOnly): self
    {
        return $this->extraParams('withAchievementsOnly', $withAchievementsOnly);
    }

    /**
     * Specifies whether only activities which unlocked gifts should be returned
     *
     * @param bool $withGiftsOnly
     * @return self
     */
    public function withGiftsOnly(bool $withGiftsOnly): self
    {
        return $this->extraParams('withGiftsOnly', $withGiftsOnly);
    }

    /**
     * Specifies whether only activities which changed KPIs should be returned
     *
     * @param bool $withKPIsOnly
     * @return self
     */
    public function withKPIsOnly(bool $withKPIsOnly): self
    {
        return $this->extraParams('withKPIsOnly', $withKPIsOnly);
    }

    /**
     * Specifies whether only compensated activities should be returned
     *
     * @param bool $withCompensatedOnly
     * @return self
     */
    public function withCompensatedOnly(bool $withCompensatedOnly): self
    {
        return $this->extraParams('withCompensatedOnly', $withCompensatedOnly);
    }

    /**
     * Specifies whether only rejected activities should be returned
     *
     * @param bool $withRejectedOnly
     * @return self
     */
    public function withRejectedOnly(bool $withRejectedOnly): self
    {
        return $this->extraParams('withRejectedOnly', $withRejectedOnly);
    }

    /**
     * Specifies whether to include behaviors with exception status
     *
     * @param bool $withExceptions
     * @return self
     */
    public function withExceptions(bool $withExceptions): self
    {
        return $this->extraParams('withExceptions', $withExceptions);
    }

    /**
     * Specifies whether the response should have nested objects populated
     *
     * @param bool $populated
     * @return self
     */
    public function populated(bool $populated): self
    {
        return $this->extraParams('populated', $populated);
    }

    /**
     * Sets the start date for filtering
     *
     * @param string $startDate
     * @return self
     */
    public function startDate(string $startDate): self
    {
        return $this->extraParams('startDate', $startDate);
    }

    /**
     * Sets the end date for filtering
     *
     * @param string $endDate
     * @return self
     */
    public function endDate(string $endDate): self
    {
        return $this->extraParams('endDate', $endDate);
    }

    /**
     * The list of reward IDs to filter by
     *
     * @param array<string> $rewardIds
     * @return self
     */
    public function rewardIds(array $rewardIds): self
    {
        return $this->extraParamsList('rewardIds', $rewardIds);
    }

    /**
     * The list of behaviour IDs to filter by
     *
     * @param array<string> $behaviourIds
     * @return self
     */
    public function behaviourIds(array $behaviourIds): self
    {
        return $this->extraParamsList('behaviourIds', $behaviourIds);
    }

    /**
     * The list of KPI IDs to filter by
     *
     * @param array<string> $kpiIds
     * @return self
     */
    public function kpiIds(array $kpiIds): self
    {
        return $this->extraParamsList('kpiIds', $kpiIds);
    }

    /**
     * The list of point IDs to filter by
     * 
     * @deprecated Point IDs are no longer supported and will be removed in a future version
     * @param array<string> $pointIds
     * @return self
     */
    public function pointIds(array $pointIds): self
    {
        return $this->extraParamsList('pointIds', $pointIds);
    }
}
