<?php
/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * This object captures the data required by the Activities API in
 * order to bulk reject activities by criteria.
 */

declare(strict_types=1);

namespace Mambo\Activity\Data;

use Mambo\Common\Data\RequestData;

class RejectActivityCriteriaRequestData implements RequestData
{
	private $data = array();

	/**
	 * The reason why the transaction was rejected. This is used to give the user
	 * insight as to why the transaction was rejected.
	 */
	public function getReason()
	{
		return $this->data['reason'];
	}
	public function setReason($reason)
	{
		$this->data['reason'] = $reason;
	}

	/**
	 * This represents the start date from which the activities should be rejected
	 * This must be a UTC timestamp in ISO 8601 format with
	 * millisecond precision: YYYY-MM-DDTHH:MM:SS.MMMZ
	 */
	public function getStartDate()
	{
		return $this->data['startDate'];
	}
	public function setStartDate($startDate)
	{
		$this->data['startDate'] = $startDate;
	}

	/**
	 * This represents the last date until which the activities should be rejected
	 * This must be a UTC timestamp in ISO 8601 format with
	 * millisecond precision: YYYY-MM-DDTHH:MM:SS.MMMZ
	 */
	public function getEndDate()
	{
		return $this->data['endDate'];
	}
	public function setEndDate($endDate)
	{
		$this->data['endDate'] = $endDate;
	}

	/**
	 * Reject activities that have a matching verb
	 */
	public function getVerb()
	{
		return $this->data['verb'];
	}
	public function setVerb($verb)
	{
		$this->data['verb'] = $verb;
	}

	/**
	 * Reject activities that have a matching UUID
	 */
	public function getUuid()
	{
		return $this->data['uuid'];
	}
	public function setUuid($uuid)
	{
		$this->data['uuid'] = $uuid;
	}

	/**
	 * Reject activities that have a matching target user UUID
	 */
	public function getTargetUserUuid()
	{
		return $this->data['targetUserUuid'];
	}
	public function setTargetUserUuid($targetUserUuid)
	{
		$this->data['targetUserUuid'] = $targetUserUuid;
	}

	/**
	 * Reject activities that have matching tag tags. Note that this will
	 * rely on the tagsJoin to determine whether we want all tags to
	 * be present or any 1 of the tags.
	 */
	public function getTagIds()
	{
		return $this->data['tagIds'];
	}
	public function setTagIds(array $tagIds)
	{
		$this->data['tagIds'] = $tagIds;
	}

	/**
	 * Reject activities that have matching tag tags. Note that this will
	 * rely on the tagsJoin to determine whether we want all tags to
	 * be present or any 1 of the tags.
	 * This method accepts the actual tag value, for example: hr_dept, finance_dept, etc.
	 */
	public function getTagTags()
	{
		return $this->data['tagTags'];
	}
	public function setTagTags(array $tagTags)
	{
		$this->data['tagTags'] = $tagTags;
	}

	/**
	 * Joins the tags provided either with hasAnyOf or hasAllOf
	 */
	public function getTagsJoin()
	{
		return $this->data['tagsJoin'];
	}
	public function setTagsJoin($tagsJoin)
	{
		$this->data['tagsJoin'] = $tagsJoin;
	}

	/**
	 * Reject activities that have matching point IDs
	 */
	public function getPointIds()
	{
		return $this->data['pointIds'];
	}
	public function setPointIds(array $pointIds)
	{
		$this->data['pointIds'] = $pointIds;
	}

	/**
	 * Reject activities that have matching reward IDs
	 */
	public function getRewardIds()
	{
		return $this->data['rewardIds'];
	}
	public function setRewardIds(array $rewardIds)
	{
		$this->data['rewardIds'] = $rewardIds;
	}

	/**
	 * Reject activities that have matching KPI IDs
	 */
	public function getKpiIds()
	{
		return $this->data['kpiIds'];
	}
	public function setKpiIds(array $kpiIds)
	{
		$this->data['kpiIds'] = $kpiIds;
	}

	/**
	 * Reject activities that have matching behaviour IDs
	 */
	public function getBehaviourIds()
	{
		return $this->data['behaviourIds'];
	}
	public function setBehaviourIds(array $behaviourIds)
	{
		$this->data['behaviourIds'] = $behaviourIds;
	}


	/**
	 * Return the JSON string equivalent of this object
	 */
	public function getJsonString(): string
	{
		return json_encode($this->data);
	}
}
