<?php
/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * Defines activity attributes specific to behaviour activities.
 */

declare(strict_types=1);

namespace Mambo\Activity\Data;

use Mambo\Common\Data\PointStore;

class ActivityBehaviourAttrs
{
	/**
	 * @var array<mixed> $data
	 */
	private $data = array();

	/**
	 * The type of attribute
	 * @return string
	 */
	public function getType()
	{
		return 'behaviour';
	}


	/**
	 * The behaviour performed by the user. Each behaviour has a corresponding verb
	 * which is the name of the behaviour in lower case. For example: The behaviour
	 * "Read an Article" would have an verb of "read an article".
	 * @return string verb
	 */
	public function getVerb()
	{
		return $this->data['verb'];
	}

	/**
	 * @param string $verb
	 * @return void
	 */
	public function setVerb($verb)
	{
		$this->data['verb'] = $verb;
	}


	/**
	 * The ID of the behaviour performed by the user. Each behaviour has a corresponding
	 * ID which can be used to track a behaviour through the API. This is an alternative
	 * to using the behaviour's verb. If verb and behaviourId are populated, the verb will
	 * take precedence.
	 * @return string behaviourId
	 */
	public function getBehaviourId()
	{
		return $this->data['behaviourId'];
	}
	public function setBehaviourId($behaviourId)
	{
		$this->data['behaviourId'] = $behaviourId;
	}


	/**
	 * The metadata associated to the behaviour performed by the user. This is an array of
	 * key/value pairs, for example: if the name of the metadata is "type" and the value is
	 * "product" then the metadata array would look like this:
	 * array( "type" => "product" )
	 * To find out more please visit the Flexible Behaviours page in the administration panel.
	 * @return string metadata
	 */
	public function getMetadata()
	{
		return $this->data['metadata'];
	}
	public function setMetadata(array $metadata)
	{
		$this->data['metadata'] = $metadata;
	}


	/**
	 * Any ad hoc points associated with this behaviour activity.
	 * The ad hoc points will be added to the points of the simple
	 * behaviour if it's status is OK. Otherwise the ad hoc points
	 * are added to the first matching flexible behaviour with a
	 * status of OK. If there are no behaviours with a status of
	 * OK, then the points are not added.
	 * @return array<PointStore>
	 */
	public function getAdhocPoints()
	{
		return $this->data['adhocPoints'];
	}
	public function setAdhocPoints(array $adhocPoints)
	{
		$this->data['adhocPoints'] = $adhocPoints;
	}
	public function addAdhocPoint(PointStore $adhocPoint)
	{
		if (!isset($this->data['adhocPoints'])) {
			$this->data['adhocPoints'] = array();
		}
		array_push($this->data['adhocPoints'], $adhocPoint);
	}


	/**
	 * Return the JSON string equivalent of this object
	 */
	public function getJsonArray()
	{
		$json = $this->data;
		$json['type'] = $this->getType();

		if (isset($json['adhocPoints'])) {
			$pointsArr = array();
			foreach ($json['adhocPoints'] as $point) {
				array_push($pointsArr, $point->getJsonArray());
			}
			$json['adhocPoints'] = $pointsArr;
		}

		return $json;
	}
}
