<?php
/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * The MamboActivitiesService class handles all Activities related requests
 * to the Mambo API.
 */

declare(strict_types=1);

namespace Mambo\Activity;

use Mambo\Activity\Data\ActivityRequestData;
use Mambo\Activity\Data\RejectActivityRequestData;
use Mambo\Activity\Data\RejectActivityCriteriaRequestData;
use Mambo\Activity\Param\ActivityGetListParams;
use Mambo\Activity\Param\ActivityGetListByUserParams;
use Mambo\Common\Service\AbstractService;
use Mambo\Http\ApiUrlBuilder;
use Mambo\Http\RequestOptions;

class ActivitiesService extends AbstractService
{
	/**
	 * Activity service end point URI
	 * @var string
	 */
	const ACTIVITIES_SITE_URI = "/v1/{site}/activities";
	const ACTIVITIES_SITE_URI_V2 = "/v2/{site}/activities";
	const ACTIVITIES_USER_URI = "/v1/{site}/activities/{uuid}";
	const ACTIVITIES_BULK_REJECT_URI = "/v1/{site}/activities/reject";
	const ACTIVITIES_ID_URI = "/v1/activities/{id}";
	const ACTIVITIES_REJECT_URI = "/v1/activities/{id}/reject";
	const ACTIVITIES_BOUNTY_CANCEL_URI = "/v1/activities/{id}/bounty/cancel";
	const ACTIVITIES_BOUNTY_AWARD_URI = "/v1/activities/{id}/bounty/award/{uuid}";
	const ACTIVITIES_STATUS_URI = "/v2/activities/{id}/status";


	/**
	 * This method is used to create activities.
	 *
	 * @deprecated This method will be removed in a future version.
	 *
	 * @param string $siteUrl					The site to which the user and behaviour belong to
	 * @param ActivityRequestData $data			The data sent to the API in order to create an activity
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function create(string $siteUrl, ActivityRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::ACTIVITIES_SITE_URI);
		$urlBuilder->withPathSite($siteUrl);
		return $this->httpPost($urlBuilder, $requestOptions, $data);
	}


	/**
	 * This method is used to create activities.
	 *
	 * @param string $siteUrl					The site to which the user and behaviour belong to
	 * @param ActivityRequestData $data			The data sent to the API in order to create an activity
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function createAsync(string $siteUrl, ActivityRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::ACTIVITIES_SITE_URI_V2);
		$urlBuilder->withPathSite($siteUrl);
		return $this->httpPost($urlBuilder, $requestOptions, $data);
	}


	/**
	 * This method is used to reject the activity with the specified ID.
	 *
	 * @param string $id							The ID of the activity to reject
	 * @param RejectActivityRequestData $data		The RejectActivityRequestData
	 * @param RequestOptions $requestOptions		The requestOptions to be used with the request
	 */
	public function reject(string $id, RejectActivityRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::ACTIVITIES_REJECT_URI);
		$urlBuilder->withPathId($id);
		return $this->httpPost($urlBuilder, $requestOptions, $data);
	}


	/**
	 * This method is used to reject activities in bulk by criteria
	 *
	 * @param string $siteUrl			The site to which the activities belong to
	 * @param RejectActivityCriteriaRequestData $data	The RejectActivityCriteriaRequestData
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function rejectBulk(string $siteUrl, RejectActivityCriteriaRequestData $data, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::ACTIVITIES_BULK_REJECT_URI);
		$urlBuilder->withPathSite($siteUrl);
		return $this->httpPost($urlBuilder, $requestOptions, $data);
	}


	/**
	 * This method is used to cancel the bounty activity with the specified ID.
	 *
	 * @param string $id		The ID of the bounty activity to cancel
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function bountyCancel(string $id, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::ACTIVITIES_BOUNTY_CANCEL_URI);
		$urlBuilder->withPathId($id);
		return $this->httpPost($urlBuilder, $requestOptions);
	}


	/**
	 * This method is used to award the bounty activity with the specified ID.
	 *
	 * @param string $id				The ID of the bounty activity to award
	 * @param string $targetUuid		The UUID to whom the bounty should be awarded
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function bountyAward(string $id, string $targetUuid, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::ACTIVITIES_BOUNTY_AWARD_URI);
		$urlBuilder->withPathId($id);
		$urlBuilder->withPathUuid($targetUuid);

		return $this->httpPost($urlBuilder, $requestOptions);
	}


	/**
	 * Get an activity by it's ID
	 *
	 * @param string $id			The ID of the activity to retrieve
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function get(string $id, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::ACTIVITIES_ID_URI);
		$urlBuilder->withPathId($id);
		return $this->httpGet($urlBuilder, $requestOptions);
	}


	/**
	 * Get the list of activities for the specified site
	 *
	 * @param string $siteUrl					The site for which to retrieve the list of activities
	 * @param ActivityGetListParams $params		The query parameters to be used with the request
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function getActivities(
		string $siteUrl,
		ActivityGetListParams $params = null,
		RequestOptions $requestOptions = null
	): array|object {

		$urlBuilder = new ApiUrlBuilder(self::ACTIVITIES_SITE_URI);
		$urlBuilder->withPathSite($siteUrl);
		$urlBuilder->withQueryParams($params);

		return $this->httpGet($urlBuilder, $requestOptions);
	}


	/**
	 * Get the list of activities for the specified user and site
	 *
	 * @param string $siteUrl						The site for which to retrieve the list of activities
	 * @param string $uuid							The user for which to retrieve the list of activities
	 * @param ActivityGetListByUserParams $params	The query parameters to be used with the request
	 * @param RequestOptions $requestOptions		The requestOptions to be used with the request
	 
	 * @deprecated The pointIds property will be removed in a future release
	 */
	public function getActivitiesByUser(
		string $siteUrl,
		string $uuid,
		ActivityGetListByUserParams $params = null,
		RequestOptions $requestOptions = null
	): array|object {

		$urlBuilder = new ApiUrlBuilder(self::ACTIVITIES_USER_URI);
		$urlBuilder->withPathSite($siteUrl);
		$urlBuilder->withPathUuid($uuid);
		$urlBuilder->withQueryParams($params);

		return $this->httpGet($urlBuilder, $requestOptions);
	}


	/**
	 * Get the processing status of a submitted Activity
	 *
	 * @param string $id			The ID of the activity to retrieve
	 * @param RequestOptions $requestOptions	The requestOptions to be used with the request
	 */
	public function getProcessingStatus(string $id, RequestOptions $requestOptions = null): array|object
	{
		$urlBuilder = new ApiUrlBuilder(self::ACTIVITIES_STATUS_URI);
		$urlBuilder->withPathId($id);
		return $this->httpGet($urlBuilder, $requestOptions);
	}
}
