/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.utils;

import static io.mambo.sdk.utils.Preconditions.checkNotNull;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;

import io.mambo.sdk.exception.MamboException;

public class StreamUtils
{
	private static final int BUFFER_SIZE = 4096;


	private StreamUtils() {}


	public static String toString( InputStream inputStream ) throws IOException
	{
		checkNotNull( inputStream, "inputStream must not be null" );

		StringBuilder stringBuilder = new StringBuilder();
		try( BufferedReader reader = toBufferedReader( inputStream ) ) {
			char[] buffer = new char[ BUFFER_SIZE ];
			int bytesRead;
			while( ( bytesRead = reader.read( buffer ) ) != -1 ) {
				stringBuilder.append( buffer, 0, bytesRead );
			}
		}

		return stringBuilder.toString();
	}


	private static BufferedReader toBufferedReader( InputStream inputStream )
	{
		return new BufferedReader(
			new InputStreamReader( inputStream, StandardCharsets.UTF_8 ) );
	}


	public static InputStream toInputStream( File file )
	{
		try {
			return Files.newInputStream( file.toPath() );
		}
		catch( IOException exception ) {
			throw new MamboException( "Error converting file to input stream", exception );
		}
	}


	public static void copy( InputStream source, OutputStream destination ) throws IOException
	{
		try {
			byte[] buffer = new byte[ 8192 ];
			int bytesRead;
			while( ( bytesRead = source.read( buffer ) ) != -1 ) {
				destination.write( buffer, 0, bytesRead );
			}
			destination.flush();
		}
		finally {
			source.close();
		}
	}
}
