/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.utils;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import lombok.EqualsAndHashCode;
import lombok.ToString;

@ToString
@EqualsAndHashCode
public class Multimap<K, V>
{
	private Map<K, List<V>> map;


	public Multimap() {
		map = MapUtils.newHashMap();
	}


	private Multimap( Map<K, List<V>> map ) {
		this.map = map;
	}


	private Multimap( Multimap<K, V> other ) {
		this.map = MapUtils.newHashMap();
		putAll( other );
	}


	public static <K, V> Multimap<K, V> of( Map<K, List<V>> map )
	{
		return new Multimap<>( map );
	}


	public static <K, V> Multimap<K, V> of( Multimap<K, V> map )
	{
		return new Multimap<>( map );
	}


	public void put( K key, V value )
	{
		map.computeIfAbsent( key, k -> ListUtils.newArrayList() ).add( value );
	}


	public void putAll( Multimap<K, V> other )
	{
		other.map.forEach( ( key, values ) -> values.forEach( value -> put( key, value ) ) );
	}


	public void putAll( K key, List<V> values )
	{
		values.forEach( value -> put( key, value ) );
	}


	public List<V> get( K key )
	{
		return map.getOrDefault( key, ListUtils.newArrayList() );
	}


	public V getFirst( K key )
	{
		List<V> values = get( key );
		return values.isEmpty() ? null : values.get( 0 );
	}


	public boolean remove( K key, V value )
	{
		List<V> values = map.get( key );
		if( values != null ) {
			boolean removed = values.remove( value );
			if( values.isEmpty() ) {
				map.remove( key );
			}
			return removed;
		}
		return false;
	}


	public List<V> removeAll( K key )
	{
		return map.remove( key );
	}


	public boolean containsKey( K key )
	{
		return map.containsKey( key );
	}


	public boolean containsValue( V value )
	{
		return map.values().stream().anyMatch( values -> values.contains( value ) );
	}


	public boolean containsEntry( K key, V value )
	{
		List<V> values = map.get( key );
		return values != null && values.contains( value );
	}


	public int size()
	{
		return map.values().stream()
			.mapToInt( List::size )
			.sum();
	}


	public boolean isEmpty()
	{
		return map.isEmpty();
	}


	public void clear()
	{
		map.clear();
	}


	public Set<K> keySet()
	{
		return map.keySet();
	}


	public List<V> values()
	{
		List<V> allValues = new ArrayList<>();
		map.values().forEach( allValues::addAll );
		return allValues;
	}


	public Set<Map.Entry<K, List<V>>> entrySet()
	{
		return map.entrySet();
	}
}