/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.utils;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.MapperFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializerProvider;
import com.fasterxml.jackson.databind.json.JsonMapper;
import com.fasterxml.jackson.databind.ser.BeanPropertyWriter;
import com.fasterxml.jackson.databind.ser.FilterProvider;
import com.fasterxml.jackson.databind.ser.PropertyWriter;
import com.fasterxml.jackson.databind.ser.impl.SimpleBeanPropertyFilter;
import com.fasterxml.jackson.databind.ser.impl.SimpleFilterProvider;

import io.mambo.sdk.exception.JsonProcessingException;
import io.mambo.sdk.http.ResponseType;
import io.mambo.sdk.service.common.model.interfaces.HasInitializedData;
import io.mambo.sdk.service.common.model.response.ResponseEntity;

/**
 * Provides a utility method to map a JSON string to a POJO.
 */
public class JsonUtils
{
	private static ObjectMapper mapper;


	private JsonUtils() {}


	static {
		mapper = buildMapper();
	}


	private static ObjectMapper buildMapper()
	{
		if( mapper != null ) {
			return mapper;
		}

		FilterProvider filterProvider = new SimpleFilterProvider().addFilter(
			"removeUninitialized",
			new HasInitializedPropertyFilter() );

    	return JsonMapper.builder()
    		.configure( MapperFeature.SORT_PROPERTIES_ALPHABETICALLY, true )
    		.configure( DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false )
    		.configure( DeserializationFeature.FAIL_ON_INVALID_SUBTYPE, false )
    		.filterProvider( filterProvider )
    		.build();
	}


	public static ObjectMapper objectMapper()
	{
		return mapper;
	}


	public static String serialize( Object toSerialise )
	{
		try {
			return mapper.writeValueAsString( toSerialise );
		} catch( Exception exception ) {
			throw new JsonProcessingException(
				"There was a problem serializing the response", toSerialise, exception );
		}
	}


	public static <T> T deserialize( String jsonString, Class<?> classType )
	{
		JavaType type = mapper.getTypeFactory().constructType( classType );
		return deserializeAsJavaType( jsonString, type );
	}


	private static <T> T deserializeAsJavaType( String jsonString, JavaType type )
	{
		try {
			return mapper.readValue( jsonString, type );
		} catch( Exception exception ) {
			throw new JsonProcessingException(
				"There was a problem deserializing the response", jsonString, exception );
		}
	}


	public static <T> T deserialize( String jsonString, Class<?> classType, ResponseType responseType )
	{
		if( responseType.isObject() ) {
			return deserialize( jsonString, classType );
		}

		if( responseType.isList() ) {
			return deserializeList( jsonString, classType );
		}

		return deserializeResponseEntity( jsonString, classType );
	}


	private static <T> T deserializeList( String jsonString, Class<?> classType )
	{
		JavaType type = mapper.getTypeFactory().constructCollectionType( ArrayList.class, classType );
		return deserializeAsJavaType( jsonString, type );
	}


	private static <T> T deserializeResponseEntity( String jsonString, Class<?> classType )
	{
		JavaType type = mapper.getTypeFactory().constructParametricType( ResponseEntity.class, classType );
		return deserializeAsJavaType( jsonString, type );
	}


    private static class HasInitializedPropertyFilter extends SimpleBeanPropertyFilter
    {
        private ThreadLocal<Set<String>> propertiesToInclude = new ThreadLocal<>();


        @Override
        protected boolean include( BeanPropertyWriter writer ) {
        	String lowerCaseProp = StringUtils.lowerCase( writer.getName() );
            return propertiesToInclude.get().contains( lowerCaseProp );
        }


        @Override
        protected boolean include( PropertyWriter writer ) {
        	String lowerCaseProp = StringUtils.lowerCase( writer.getName() );
            return propertiesToInclude.get().contains( lowerCaseProp );
        }


        @Override
        public void serializeAsField(Object pojo, JsonGenerator jgen,
                SerializerProvider provider, PropertyWriter writer)
            throws Exception
        {
    		propertiesToInclude.remove();
    		propertiesToInclude.set( new HashSet<>() );

        	if( !HasInitializedData.class.isAssignableFrom( pojo.getClass() ) ) {
            	super.serializeAsField( pojo, jgen, provider, writer );
            	return;
        	}

    		HasInitializedData data = (HasInitializedData) pojo;

    		if( ListUtils.isEmpty( data.getInitializedFields() ) ) {
            	super.serializeAsField( pojo, jgen, provider, writer );
            	return;
    		}

    		List<String> lowerCasePropertiesToInclude = data.getInitializedFields().stream()
    				.map( StringUtils::lowerCase )
    				.collect( Collectors.toCollection( ArrayList::new ) );

    		propertiesToInclude.get().addAll( lowerCasePropertiesToInclude );
        	super.serializeAsField( pojo, jgen, provider, writer );
        }
    }
}
