/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.webhook.model;

import java.util.ArrayList;
import java.util.List;

import io.mambo.sdk.service.common.model.interfaces.HasInitializedData;
import io.mambo.sdk.service.webhook.data.WebhookRequestData;
import lombok.EqualsAndHashCode;
import lombok.ToString;

/**
 * Webhook data shared by {@link WebhookDto} and {@link WebhookRequestData}
 */
@ToString
@EqualsAndHashCode
public abstract class AbstractWebhook implements HasInitializedData
{
	protected transient List<String> initializedFields = new ArrayList<>();

	private String name;
	private String description;
	private String endpointUrl;
	private String secret;
	private Boolean active;
	private Boolean sslVerification;
	private List<String> events;
	private List<String> siteIds;
	private String payloadVersion;


	/**
	 * The name of the webhook.
	 * See the webhook page in administration panel for more information.
	 * @return
	 */
	public String getName() { return name; }
	public void setName( String name ) {
		initializedFields.add( "name" );
		this.name = name;
	}


	/**
	 * The description for the webhook
	 * See the webhook page in administration panel for more information.
	 * @return
	 */
	public String getDescription() { return description; }
	public void setDescription( String description ) {
		initializedFields.add( "description" );
		this.description = description;
	}


	/**
	 * Indicates whether the webhook is active or inactive
	 * See the webhook page in administration panel for more information.
	 * @return
	 */
	public Boolean getActive() { return active; }
	public void setActive( Boolean active ) {
		initializedFields.add( "active" );
		this.active = active;
	}


	/**
	 * The endpoint URL to which the webhook payload should be sent
	 * See the webhook page in administration panel for more information.
	 * @return
	 */
	public String getEndpointUrl() { return endpointUrl; }
	public void setEndpointUrl( String endpointUrl ) {
		initializedFields.add( "endpointUrl" );
		this.endpointUrl = endpointUrl;
	}


	/**
	 * Indicates whether the SSL certificates of the endpoint
	 * should be verified before sending the payload
	 * See the webhook page in administration panel for more information.
	 * @return
	 */
	public Boolean getSslVerification() { return sslVerification; }
	public void setSslVerification( Boolean sslVerification ) {
		initializedFields.add( "sslVerification" );
		this.sslVerification = sslVerification;
	}


	/**
	 * The secret is used to sign the payload to ensure it comes from Mambo
	 * See the webhook page in administration panel for more information.
	 * @return
	 */
	public String getSecret() { return secret; }
	public void setSecret( String secret ) {
		initializedFields.add( "secret" );
		this.secret = secret;
	}


	/**
	 * The events which should cause the webhook to deliver the payload.
	 * Valid values include: ActivityCreated, ActivityBehavioursAdded, ActivityAwardedAdded, ,
	 * ActivityRejected, NotificationCreated, NotificationRead, PointCreated, PointUpdated, PointWalletDebitExpired,
	 * BehaviourCreated, BehaviourUpdated, RewardCreated, RewardUpdated, KPICreated, KPIUpdated, LeaderboardCreated,
	 * LeaderboardUpdated, LeaderboardRankingUpdated, CouponCreated, CouponUpdated, TagCreated, TagUpdated,
	 * UserCreated, UserUpdated, UserMemberJoined, UserMemberLeft
	 * See the webhook page in administration panel for more information.
	 * @return
	 */
	public List<String> getEvents() { return events; }
	public void setEvents( List<String> events ) {
		initializedFields.add( "events" );
		this.events = events;
	}


	/**
	 * The sites from which the webhook should listen for events
	 * See the webhook page in administration panel for more information.
	 * @return
	 */
	public List<String> getSiteIds() { return siteIds; }
	public void setSiteIds( List<String> siteIds ) {
		initializedFields.add( "siteIds" );
		this.siteIds = siteIds;
	}


	/**
	 * Whether Webhook payloads should use the pinned version of the latest
	 * version of Mambo. Valid values are "pinned" and "latest".
	 * See the webhook section in the administrator manual for more information.
	 * @return
	 */
	public String getPayloadVersion() { return payloadVersion; }
	public void setPayloadVersion( String payloadVersion ) {
		initializedFields.add( "payloadVersion" );
		this.payloadVersion = payloadVersion;
	}


	@Override
	public List<String> getInitializedFields() {
		return initializedFields;
	}
}
