/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.webhook;

import io.mambo.sdk.http.HttpMethod;
import io.mambo.sdk.http.RequestOptions;
import io.mambo.sdk.http.ResponseType;
import io.mambo.sdk.http.api.ApiRequest;
import io.mambo.sdk.http.api.ApiRequestAdapter;
import io.mambo.sdk.service.common.AbstractService;
import io.mambo.sdk.service.common.data.DeleteRequestData;
import io.mambo.sdk.service.common.model.response.ResponseEntity;
import io.mambo.sdk.service.common.model.response.Status;
import io.mambo.sdk.service.webhook.data.WebhookRequestData;
import io.mambo.sdk.service.webhook.model.WebhookDto;
import io.mambo.sdk.service.webhook.param.WebhookDeleteParams;
import io.mambo.sdk.service.webhook.param.WebhookGetListParams;
import io.mambo.sdk.service.webhook.param.WebhookGetParams;
import io.mambo.sdk.service.webhook.param.WebhookUpdateParams;

/**
 * The WebhooksService class handles all Webhook related requests to the Mambo
 * API.
 */
public class WebhooksService extends AbstractService
{
	private static final String WEBHOOKS_URI = "/v1/webhooks";
	private static final String WEBHOOKS_ID_URI = WEBHOOKS_URI + "/{webhookId}";


	public WebhooksService( ApiRequestAdapter apiClient ) {
		super( apiClient );
	}


	/**
	 * This method is used to create a new webhook.
	 *
	 * @see WebhookRequestData
	 *
	 * @param data
	 *            The webhook request data
	 * @return
	 */
	public WebhookDto create( WebhookRequestData data )
	{
		return create( data, RequestOptions.create() );
	}


	/**
	 * This method is used to create a new webhook.
	 *
	 * @see WebhookRequestData
	 *
	 * @param data
	 *            The webhook request data
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public WebhookDto create( WebhookRequestData data, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( WEBHOOKS_URI )
			.responseClass( WebhookDto.class )
			.method( HttpMethod.POST )
			.requestData( data )
			.options( requestOptions )
			.build() );
	}


	/**
	 * Update an existing webhook by ID.
	 *
	 * @param webhookId
	 *            The ID of the webhook to update
	 * @param data
	 *            The data with which to update the specified webhook object
	 * @return
	 */
	public WebhookDto update( String webhookId, WebhookRequestData data )
	{
		return update( WebhookUpdateParams.builder()
			.webhookId( webhookId )
			.build(), data );
	}


	/**
	 * Update an existing webhook by ID.
	 *
	 * @param params
	 *            The parameters required to update the webhook
	 * @param data
	 *            The data with which to update the specified webhook object
	 * @return
	 */
	public WebhookDto update( WebhookUpdateParams params, WebhookRequestData data )
	{
		return update( params, data, RequestOptions.create() );
	}


	/**
	 * Update an existing webhook by ID.
	 *
	 * @param params
	 *            The parameters required to update the webhook
	 * @param data
	 *            The data with which to update the specified webhook object
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public WebhookDto update(
		WebhookUpdateParams params,
		WebhookRequestData data,
		RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( WEBHOOKS_ID_URI )
			.responseClass( WebhookDto.class )
			.method( HttpMethod.PUT )
			.requestData( data )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Delete a webhook by ID.
	 *
	 * @param webhookId
	 *            The ID of the webhook to delete
	 * @return
	 */
	public Status delete( String webhookId )
	{
		return delete( WebhookDeleteParams.builder()
			.webhookId( webhookId )
			.build() );
	}


	/**
	 * Delete a webhook by ID.
	 *
	 * @param params
	 *            The parameters required to delete the webhook
	 * @return
	 */
	public Status delete( WebhookDeleteParams params )
	{
		return delete( params, RequestOptions.create() );
	}


	/**
	 * Delete a webhook by ID.
	 *
	 * @param params
	 *            The parameters required to delete the webhook
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public Status delete( WebhookDeleteParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( WEBHOOKS_ID_URI )
			.responseClass( Status.class )
			.method( HttpMethod.DELETE )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Get a webhook by ID
	 *
	 * @param webhookId
	 *            The ID of the webhook to retrieve
	 * @return
	 */
	public WebhookDto get( String webhookId )
	{
		return get( WebhookGetParams.builder()
			.webhookId( webhookId )
			.build() );
	}


	/**
	 * Get a webhook by ID
	 *
	 * @param params
	 *            The parameters required to retrieve the webhook
	 * @return
	 */
	public WebhookDto get( WebhookGetParams params )
	{
		return get( params, RequestOptions.create() );
	}


	/**
	 * Get a webhook by ID
	 *
	 * @param params
	 *            The parameters required to retrieve the webhook
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public WebhookDto get( WebhookGetParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( WEBHOOKS_ID_URI )
			.responseClass( WebhookDto.class )
			.method( HttpMethod.GET )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Delete a list of webhooks by ID
	 *
	 * @param data
	 *            The {@link DeleteRequestData}
	 * @return
	 */
	public Status deleteWebhooks( DeleteRequestData data )
	{
		return deleteWebhooks( data, RequestOptions.create() );
	}


	/**
	 * Delete a list of webhooks by ID
	 *
	 * @param data
	 *            The {@link DeleteRequestData}
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public Status deleteWebhooks( DeleteRequestData data, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( WEBHOOKS_URI )
			.responseClass( Status.class )
			.method( HttpMethod.DELETE )
			.requestData( data )
			.options( requestOptions )
			.build() );
	}


	/**
	 * Get the list of webhooks for the specified site
	 *
	 * @return
	 */
	public ResponseEntity<WebhookDto> getWebhooks()
	{
		return getWebhooks( WebhookGetListParams.builder().build() );
	}


	/**
	 * Get the list of webhooks
	 *
	 * @param params
	 *            The parameters required to retrieve a list of webhooks
	 * @return
	 */
	public ResponseEntity<WebhookDto> getWebhooks( WebhookGetListParams params )
	{
		return getWebhooks( params, RequestOptions.create() );
	}


	/**
	 * Get the list of webhooks
	 *
	 * @param params
	 *            The parameters required to retrieve a list of webhooks
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public ResponseEntity<WebhookDto> getWebhooks(
		WebhookGetListParams params,
		RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( WEBHOOKS_URI )
			.responseClass( WebhookDto.class )
			.responseType( ResponseType.RESPONSE_ENTITY )
			.method( HttpMethod.GET )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Return an empty {@link WebhookRequestData} object
	 *
	 * @return
	 */
	public WebhookRequestData newWebhookRequestData()
	{
		return new WebhookRequestData();
	}
}