/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.user.param.activity;

import static io.mambo.sdk.utils.Preconditions.checkContainsKey;
import static io.mambo.sdk.utils.Preconditions.checkNotNull;

import java.util.List;

import io.mambo.sdk.service.common.param.AbstractServiceParamsBuilder;
import io.mambo.sdk.service.common.param.ServiceParams;
import io.mambo.sdk.utils.Multimap;

/**
 * Parameters for getting user activities
 */
public class UserGetActivitiesParams implements ServiceParams
{
	private Multimap<String, String> params = new Multimap<>();


	private UserGetActivitiesParams( Builder builder ) {
		this.params = builder.params();
	}


	public static Builder builder()
	{
		return new Builder();
	}


	@Override
	public Multimap<String, String> asMap()
	{
		return params;
	}


	public static class Builder extends AbstractServiceParamsBuilder<Builder>
	{
		private static final String UUID_MUST_NOT_BE_NULL = "uuid must not be null";
		private static final String SITE_URL_MUST_NOT_BE_NULL = "siteUrl must not be null";


		/**
		 * The unique ID of the user whose activities to get
		 */
		public Builder uuid( String uuid )
		{
			checkNotNull( uuid, UUID_MUST_NOT_BE_NULL );
			params.put( "uuid", uuid );
			return this;
		}


		/**
		 * The site URL where the user exists
		 */
		public Builder siteUrl( String siteUrl )
		{
			checkNotNull( siteUrl, SITE_URL_MUST_NOT_BE_NULL );
			params.put( "siteUrl", siteUrl );
			return this;
		}


		/**
		 * Specifies the page of results to retrieve.
		 */
		public Builder page( int page )
		{
			params.put( "page", String.valueOf( page ) );
			return this;
		}


		/**
		 * Specifies the number of results to retrieve, up to a maximum of 100. Check
		 * the documentation to see which APIs support this option
		 */
		public Builder count( int count )
		{
			params.put( "count", String.valueOf( count ) );
			return this;
		}


		/**
		 * The list of tags to filter by
		 */
		public Builder tags( List<String> tags )
		{
			params.putAll( "tags", tags );
			return this;
		}


		/**
		 * Whether tags should filter for any or all tags
		 */
		public Builder tagsJoin( String tagsJoin )
		{
			params.put( "tagsJoin", tagsJoin );
			return this;
		}


		/**
		 * Whether to filter by personalization tags
		 */
		public Builder withPersonalization( boolean withPersonalization )
		{
			params.put( "withPersonalization", String.valueOf( withPersonalization ) );
			return this;
		}


		/**
		 * Specifies whether only activities which unlocked one or more behaviours
		 * should be returned.
		 */
		public Builder withBehavioursOnly( boolean withBehavioursOnly )
		{
			params.put( "withBehavioursOnly", String.valueOf( withBehavioursOnly ) );
			return this;
		}


		/**
		 * Specifies whether only activities which unlocked one or more rewards should
		 * be returned.
		 */
		public Builder withRewardsOnly( boolean withRewardsOnly )
		{
			params.put( "withRewardsOnly", String.valueOf( withRewardsOnly ) );
			return this;
		}


		/**
		 * Specifies whether only activities which unlocked one or more missions should
		 * be returned.
		 */
		public Builder withMissionsOnly( boolean withMissionsOnly )
		{
			params.put( "withMissionsOnly", String.valueOf( withMissionsOnly ) );
			return this;
		}


		/**
		 * Specifies whether only activities which unlocked one or more levels should be
		 * returned.
		 */
		public Builder withLevelsOnly( boolean withLevelsOnly )
		{
			params.put( "withLevelsOnly", String.valueOf( withLevelsOnly ) );
			return this;
		}


		/**
		 * Specifies whether only activities which unlocked one or more achievements
		 * should be returned.
		 */
		public Builder withAchievementsOnly( boolean withAchievementsOnly )
		{
			params.put( "withAchievementsOnly", String.valueOf( withAchievementsOnly ) );
			return this;
		}


		/**
		 * Specifies whether only activities which unlocked one or more gifts should be
		 * returned.
		 */
		public Builder withGiftsOnly( boolean withGiftsOnly )
		{
			params.put( "withGiftsOnly", String.valueOf( withGiftsOnly ) );
			return this;
		}


		/**
		 * Specifies whether only activities which changed one or more KPIs should be
		 * returned.
		 */
		public Builder withKPIsOnly( boolean withKPIsOnly )
		{
			params.put( "withKPIsOnly", String.valueOf( withKPIsOnly ) );
			return this;
		}


		/**
		 * Specifies whether only activities which were compensated should be returned
		 */
		public Builder withCompensatedOnly( boolean withCompensatedOnly )
		{
			params.put( "withCompensatedOnly", String.valueOf( withCompensatedOnly ) );
			return this;
		}


		/**
		 * Specifies whether only activities which have been rejected should be
		 * returned.
		 */
		public Builder withRejectedOnly( boolean withRejectedOnly )
		{
			params.put( "withRejectedOnly", String.valueOf( withRejectedOnly ) );
			return this;
		}


		/**
		 * Specifies whether the behaviour activities returned should include behaviours
		 * which have an exception status.
		 */
		public Builder withExceptions( boolean withExceptions )
		{
			params.put( "withExceptions", String.valueOf( withExceptions ) );
			return this;
		}


		/**
		 * Include activities with matching target user
		 */
		public Builder withTargetUser( boolean withTargetUser )
		{
			params.put( "withTargetUser", String.valueOf( withTargetUser ) );
			return this;
		}


		/**
		 * Specifies whether the response should have nested objects populated.
		 */
		public Builder populated( boolean populated )
		{
			params.put( "populated", String.valueOf( populated ) );
			return this;
		}


		/**
		 * Sets the start date for filtering
		 */
		public Builder startDate( String startDate )
		{
			params.put( "startDate", startDate );
			return this;
		}


		/**
		 * Sets the end date for filtering
		 */
		public Builder endDate( String endDate )
		{
			params.put( "endDate", endDate );
			return this;
		}


		/**
		 * The list of point IDs to filter by (if any)
		 *
		 * @deprecated
		 */
		@Deprecated
		public Builder pointIds( List<String> pointIds )
		{
			params.putAll( "pointIds", pointIds );
			return this;
		}


		/**
		 * The list of reward IDs to filter by (if any)
		 */
		public Builder rewardIds( List<String> rewardIds )
		{
			params.putAll( "rewardIds", rewardIds );
			return this;
		}


		/**
		 * The list of behaviour IDs to filter by (if any)
		 */
		public Builder behaviourIds( List<String> behaviourIds )
		{
			params.putAll( "behaviourIds", behaviourIds );
			return this;
		}


		/**
		 * The list of KPI IDs to filter by (if any)
		 */
		public Builder kpiIds( List<String> kpiIds )
		{
			params.putAll( "kpiIds", kpiIds );
			return this;
		}


		public UserGetActivitiesParams build()
		{
			checkContainsKey( params, "uuid", UUID_MUST_NOT_BE_NULL );
			checkContainsKey( params, "siteUrl", SITE_URL_MUST_NOT_BE_NULL );
			return new UserGetActivitiesParams( this );
		}


		private Multimap<String, String> params()
		{
			return params;
		}
	}
}
