/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.user.param;

import static io.mambo.sdk.utils.Preconditions.checkContainsKey;
import static io.mambo.sdk.utils.Preconditions.checkNotNull;

import java.util.List;

import io.mambo.sdk.service.common.param.AbstractServiceParamsBuilder;
import io.mambo.sdk.service.common.param.ServiceParams;
import io.mambo.sdk.utils.Multimap;

/**
 * Parameters for updating an existing user
 */
public class UserUpdateParams implements ServiceParams
{
	private Multimap<String, String> params = new Multimap<>();


	private UserUpdateParams( Builder builder ) {
		this.params = builder.params();
	}


	public static Builder builder()
	{
		return new Builder();
	}


	@Override
	public Multimap<String, String> asMap()
	{
		return params;
	}


	public static class Builder extends AbstractServiceParamsBuilder<Builder>
	{
		private static final String UUID_MUST_NOT_BE_NULL = "uuid must not be null";
		private static final String SITE_URL_MUST_NOT_BE_NULL = "siteUrl must not be null";


		/**
		 * The unique ID of the user to update
		 */
		public Builder uuid( String uuid )
		{
			checkNotNull( uuid, UUID_MUST_NOT_BE_NULL );
			params.put( "uuid", uuid );
			return this;
		}


		/**
		 * The site URL where the user exists
		 */
		public Builder siteUrl( String siteUrl )
		{
			checkNotNull( siteUrl, SITE_URL_MUST_NOT_BE_NULL );
			params.put( "siteUrl", siteUrl );
			return this;
		}


		/**
		 * Whether to filter by personalization tags
		 */
		public Builder withPersonalization( boolean withPersonalization )
		{
			params.put( "withPersonalization", String.valueOf( withPersonalization ) );
			return this;
		}


		/**
		 * User properties to return
		 */
		public Builder properties( List<String> properties )
		{
			params.putAll( "props", properties );
			return this;
		}


		public UserUpdateParams build()
		{
			checkContainsKey( params, "uuid", UUID_MUST_NOT_BE_NULL );
			checkContainsKey( params, "siteUrl", SITE_URL_MUST_NOT_BE_NULL );
			return new UserUpdateParams( this );
		}


		private Multimap<String, String> params()
		{
			return params;
		}
	}
}
