/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.user.model;

import java.util.List;

import io.mambo.sdk.service.common.model.ImageDto;
import io.mambo.sdk.service.common.model.awarded.AwardedRewardDto;
import io.mambo.sdk.service.common.model.interfaces.AggregateDto;
import io.mambo.sdk.service.common.model.interfaces.HasId;
import io.mambo.sdk.service.common.model.tag.HasTags;
import io.mambo.sdk.service.tag.model.TagDto;
import io.mambo.sdk.utils.StringUtils;
import lombok.EqualsAndHashCode;
import lombok.ToString;

/**
 * Provides the information associated with a specific User.
 * The User object will include the last level and achievement
 * the user unlocked as well as the total number of points.
 */
@SuppressWarnings( "deprecation" )
@ToString( callSuper = true )
@EqualsAndHashCode( callSuper = true )
public class UserDto extends AbstractUser implements AggregateDto, HasTags, HasId
{
	private String id;
	private String createdOn;
	private String lastSeenOn;
	private String memberSince;
	private String lastActivityOn;
	private ImageDto image = new ImageDto();
	private List<UserPointsDto> points;
	private AwardedRewardDto level;
	private AwardedRewardDto lastReward;
	private UserStatsDto stats;
	private Boolean blacklisted;
	private List<TagDto> tags;
	private List<String> tagIds;

	/**
	 * This ID uniquely identifies this object
	 */
	@Override
	public String getId() { return id; }
	public void setId( String id ) { this.id = id; }

	/**
	 * The User's image. This is an object with the data.
	 * @return
	 */
	public ImageDto getImage() { return image; }
	public void setImage( ImageDto image ) { this.image = image; }


	/**
	 * The time and date when the user was created.
	 * Example value: 2023-02-01T23:45:01.000Z
	 * @return
	 */
	public String getCreatedOn() { return createdOn; }
	public void setCreatedOn( String createdOn ) { this.createdOn = createdOn; }

	/**
	 * This is the last date on which any user data model changes have been recorded.
	 * Example value: 2023-02-01T23:45:01.000Z
	 * @return
	 */
	public String getLastSeenOn() { return lastSeenOn; }
	public void setLastSeenOn( String lastSeenOn ) { this.lastSeenOn = lastSeenOn; }

	/**
	 * This is the last date on which any user activity has been recorded.
	 * Example value: 2023-02-01T23:45:01.000Z
	 * @return
	 */
	public String getLastActivityOn() { return lastActivityOn; }
	public void setLastActivityOn( String lastActivityOn ) { this.lastActivityOn = lastActivityOn; }

	/**
	 * The time and date when the user opted into being a member of the rewards program
	 * Example value: 2023-02-01T23:45:01.000Z
	 * @return
	 */
	public String getMemberSince() { return memberSince; }
	public void setMemberSince( String memberSince ) { this.memberSince = memberSince; }

	/**
	 * The points the user has won by performing behaviours.
	 * @return
	 */
	public List<UserPointsDto> getPoints() { return points; }
	public void setPoints( List<UserPointsDto> points ) { this.points = points; }
	public UserPointsDto getPoint( String pointId )
	{
		if( points == null ) {
			return null;
		}

		for( UserPointsDto point : points )
		{
			if( StringUtils.equals( point.getPointId(), pointId ) ) {
				return point;
			}
		}
		return null;
	}

	/**
	 * Whether the user has been blacklisted.
	 * @return
	 */
	public Boolean isBlacklisted() { return blacklisted; }
	public void setBlacklisted( Boolean blacklisted ) { this.blacklisted = blacklisted; }

	/**
	 * The last level unlocked by the user.
	 * @return
	 */
	public AwardedRewardDto getLevel() { return level; }
	public void setLevel( AwardedRewardDto level ) { this.level = level; }

	/**
	 * The last achievement unlocked by the user.
	 * @return
	 */
	public AwardedRewardDto getLastReward() { return lastReward; }
	public void setLastReward( AwardedRewardDto lastReward ) { this.lastReward = lastReward; }

	/**
	 * The user's stats.
	 * @return
	 */
	public UserStatsDto getStats() { return stats; }
	public void setStats( UserStatsDto stats ) { this.stats = stats; }

	/**
	 * This should contain the list of the IDs of the tags which
	 * must be added to the object.
	 * @return
	 */
	@Override
	public List<String> getTagIds() { return tagIds; }
	@Override
	public void setTagIds( List<String> tagIds ) {
		this.tagIds = tagIds;
	}


	/**
	 * This contains the list of {@link TagDto} associated to this object.
	 * @return
	 */
	@Override
	public List<TagDto> getTags() { return tags; }
	@Override
	public void setTags( List<TagDto> tags ) {
		this.tags = tags;
	}
}
